/**
 * Postal Code Auto-Fill System
 * Automatically fills postal code based on selected city, state, and country
 * Uses Google Geocoding API for accurate postal code retrieval
 */

class PostalCodeAutoFill {
    constructor(options = {}) {
        this.countrySelector = options.countrySelector || '#country';
        this.stateSelector = options.stateSelector || '#state';
        this.citySelector = options.citySelector || '#city';
        this.postalCodeInput = options.postalCodeInput || '#postal_code';
        this.apiKeyEndpoint = options.apiKeyEndpoint || '/api/get-google-maps-key';
        this.apiKey = null;
        this.cache = new Map(); // Cache postal codes to reduce API calls
        
        this.init();
    }

    async init() {
        try {
            // Fetch Google Maps API key
            const response = await fetch(this.apiKeyEndpoint);
            const data = await response.json();
            
            if (data.success && data.api_key) {
                this.apiKey = data.api_key;
                this.attachEventListeners();
            } else {
                console.warn('Google Maps API key not configured. Postal code auto-fill disabled.');
            }
        } catch (error) {
            console.error('Failed to initialize postal code auto-fill:', error);
        }
    }

    attachEventListeners() {
        const self = this;
        
        // Listen for city selection changes
        $(document).on('change', this.citySelector, function() {
            const city = $(this).find('option:selected').text();
            const state = $(self.stateSelector).find('option:selected').text();
            const country = $(self.countrySelector).find('option:selected').text();
            
            if (city && city !== 'Select' && state && state !== 'Select' && country && country !== 'Select') {
                self.fetchPostalCode(city, state, country);
            }
        });
    }

    async fetchPostalCode(city, state, country) {
        if (!this.apiKey) {
            console.warn('API key not available');
            return;
        }

        // Check cache first
        const cacheKey = `${country}-${state}-${city}`.toLowerCase();
        if (this.cache.has(cacheKey)) {
            this.updatePostalCode(this.cache.get(cacheKey));
            return;
        }

        // Try multiple address formats to get better results
        const addressFormats = [
            `${city}, ${state}, ${country}`,
            `${city} ${state} ${country}`, // Without commas
            `${city}, ${country}`,
            `${city} Central, ${state}, ${country}`, // Try city center
            `${city} CBD, ${state}, ${country}` // Try CBD (Central Business District)
        ];

        for (const address of addressFormats) {
            const geocodeUrl = `https://maps.googleapis.com/maps/api/geocode/json?address=${encodeURIComponent(address)}&key=${this.apiKey}&region=${this.getCountryCode(country)}`;

            try {
                const response = await fetch(geocodeUrl);
                const data = await response.json();

                if (data.status === 'OK' && data.results.length > 0) {
                    // Try all results, not just the first one
                    for (const result of data.results) {
                        // Extract postal code from address components
                        // Try multiple postal code types
                        const postalCodeComponent = result.address_components.find(component => 
                            component.types.includes('postal_code') || 
                            component.types.includes('postal_code_prefix')
                        );

                        if (postalCodeComponent) {
                            const postalCode = postalCodeComponent.long_name;
                            
                            // Cache the result
                            this.cache.set(cacheKey, postalCode);
                            
                            // Update the postal code input
                            this.updatePostalCode(postalCode);
                            return; // Success, exit
                        }
                    }
                }
            } catch (error) {
                console.error('Error fetching postal code:', error);
            }
        }

        // If we get here, no postal code was found after trying all formats
        console.log('No postal code found after trying multiple address formats for:', city, state, country);
        
        // Show a subtle message to the user
        const $postalInput = $(this.postalCodeInput);
        if (!$postalInput.val()) {
            $postalInput.attr('placeholder', 'Enter postal code manually');
        }
    }

    updatePostalCode(postalCode) {
        const $postalInput = $(this.postalCodeInput);
        
        // Only auto-fill if the field is empty or user hasn't manually entered a value
        if (!$postalInput.val() || !$postalInput.data('user-modified')) {
            $postalInput.val(postalCode);
            $postalInput.removeClass('is-invalid');
            
            // Show a subtle indicator that it was auto-filled
            $postalInput.addClass('auto-filled');
            setTimeout(() => {
                $postalInput.removeClass('auto-filled');
            }, 1500);
        }
    }

    // Get country code for better geocoding results
    getCountryCode(countryName) {
        const countryCodes = {
            'South Africa': 'za',
            'United States': 'us',
            'United Kingdom': 'gb',
            'Canada': 'ca',
            'Australia': 'au',
            'India': 'in',
            'Germany': 'de',
            'France': 'fr',
            'Italy': 'it',
            'Spain': 'es',
            'Netherlands': 'nl',
            'Belgium': 'be',
            'Switzerland': 'ch',
            'Austria': 'at',
            'Sweden': 'se',
            'Norway': 'no',
            'Denmark': 'dk',
            'Finland': 'fi',
            'Poland': 'pl',
            'Portugal': 'pt',
            'Greece': 'gr',
            'Ireland': 'ie',
            'New Zealand': 'nz',
            'Singapore': 'sg',
            'Malaysia': 'my',
            'Thailand': 'th',
            'Philippines': 'ph',
            'Indonesia': 'id',
            'Vietnam': 'vn',
            'Japan': 'jp',
            'South Korea': 'kr',
            'China': 'cn',
            'Brazil': 'br',
            'Mexico': 'mx',
            'Argentina': 'ar',
            'Chile': 'cl',
            'Colombia': 'co',
            'Peru': 'pe',
            'Egypt': 'eg',
            'Nigeria': 'ng',
            'Kenya': 'ke',
            'Ghana': 'gh',
            'Morocco': 'ma',
            'Algeria': 'dz',
            'Tunisia': 'tn',
            'United Arab Emirates': 'ae',
            'Saudi Arabia': 'sa',
            'Israel': 'il',
            'Turkey': 'tr',
            'Russia': 'ru',
            'Ukraine': 'ua',
            'Czech Republic': 'cz',
            'Hungary': 'hu',
            'Romania': 'ro',
            'Bulgaria': 'bg'
        };
        
        return countryCodes[countryName] || '';
    }

    // Allow manual input to override auto-fill
    markAsUserModified() {
        const self = this;
        $(this.postalCodeInput).on('input', function() {
            $(this).data('user-modified', true);
        });
    }
}

// Initialize multiple instances for different forms
$(document).ready(function() {
    // Add CSS for auto-fill indicator
    if (!$('#postal-autofill-styles').length) {
        $('head').append(`
            <style id="postal-autofill-styles">
                .auto-filled {
                    border-color: #28a745 !important;
                    transition: border-color 0.3s ease;
                }
            </style>
        `);
    }

    // Function to initialize all forms
    function initializeAllForms() {
        const formConfigs = [
            // Provider Profile
            {
                countrySelector: '#country',
                stateSelector: '#state',
                citySelector: '#city',
                postalCodeInput: '#postal_code'
            },
            // Provider Registration (modal) - Home popup
            {
                countrySelector: '#provider_country',
                stateSelector: '#provider_state',
                citySelector: '#provider_city',
                postalCodeInput: '#provider_postal_code'
            },
            // User Registration (modal) - Home popup "Join Us"
            {
                countrySelector: '#user_country',
                stateSelector: '#user_state',
                citySelector: '#city',
                postalCodeInput: '#postal_code'
            },
            // Admin/Staff forms
            {
                countrySelector: '.country',
                stateSelector: '.state',
                citySelector: '.city',
                postalCodeInput: '#postal_code'
            },
            // Edit forms
            {
                countrySelector: '#edit_country',
                stateSelector: '#edit_state',
                citySelector: '#edit_city',
                postalCodeInput: '#edit_postal_code'
            },
            // Branch forms (add & edit use different IDs)
            {
                countrySelector: '#country',
                stateSelector: '#state',
                citySelector: '#city',
                postalCodeInput: '#zip_code'
            }
        ];

        // Initialize postal code auto-fill for each form
        formConfigs.forEach(config => {
            // Check if the selectors exist before initializing
            if ($(config.citySelector).length > 0) {
                new PostalCodeAutoFill(config);
            }
        });
    }

    // Initialize immediately for existing forms
    initializeAllForms();

    // Re-initialize when modals are shown (for dynamically loaded content)
    $('#provider').on('shown.bs.modal', function() {
        setTimeout(function() {
            // Initialize for provider registration form
            if ($('#provider_city').length > 0) {
                new PostalCodeAutoFill({
                    countrySelector: '#provider_country',
                    stateSelector: '#provider_state',
                    citySelector: '#provider_city',
                    postalCodeInput: '#provider_postal_code'
                });
            }
        }, 500);
    });

    $('#register-modal').on('shown.bs.modal', function() {
        setTimeout(function() {
            // Initialize for user registration form
            if ($('#city').length > 0 && $('#register-modal').is(':visible')) {
                new PostalCodeAutoFill({
                    countrySelector: '#user_country',
                    stateSelector: '#user_state',
                    citySelector: '#city',
                    postalCodeInput: '#postal_code'
                });
            }
        }, 500);
    });
});

// Export for manual initialization if needed
window.PostalCodeAutoFill = PostalCodeAutoFill;

