// Load active languages for multi-select dropdown
function loadActiveLanguages() {
    $.ajax({
        url: "/api/admin/languages",
        type: "POST",
        headers: {
            "Accept": "application/json",
            "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr("content"),
        },
        data: {},
        success: function (response) {
            console.log("Language API response:", response);
            
            if (response.code === 200 && response.data) {
                const languageSelect = $("#provider_language");
                languageSelect.empty();
                
                // Response.data is the array directly, not response.data.data
                response.data.forEach(function (language) {
                    if (language.status === 1) { // Only active languages
                        languageSelect.append(
                            `<option value="${language.id}">${language.name}</option>`
                        );
                    }
                });
                
                // Initialize Select2 for multi-select
                languageSelect.select2({
                    placeholder: "Select languages you speak",
                    allowClear: true,
                    width: '100%'
                });
                
                console.log("Languages loaded successfully:", response.data.length);
            } else {
                console.error("Unexpected response structure:", response);
            }
        },
        error: function (xhr) {
            console.error("Failed to load languages:", xhr);
            console.error("Response text:", xhr.responseText);
        }
    });
}

$(document).ready(function () {
    $("#categorySelect").on("change", function () {
        $(this).valid();
        const categoryId = $(this).val();
        var langCode = $("body").data("lang");
        fetchSubcategories(categoryId, langCode);
    });
});

$(document).ready(function () {
    $("#provider").on("hidden.bs.modal", function () {
        // Reset the form fields
        $("#providerRegister")[0].reset();

        // Remove validation states
        $(".form-control").removeClass("is-invalid").removeClass("is-valid");
        $(".invalid-feedback").text("");

        // Show the first fieldset and hide others
        $(".wizard-fieldset fieldset").hide(); // Hide all fieldsets
        $(".first-field").show(); // Show the first fieldset
    });
});

// Password toggle - check if element exists (might be on page or in modal)
const providerTogglePassword = document.getElementById("providerTogglePassword");
if (providerTogglePassword) {
    providerTogglePassword.addEventListener("click", function () {
        const passwordField = document.getElementById("provider_password");
        const toggleIcon = document.getElementById("toggleIcon");

        if (passwordField && toggleIcon) {
            const isPassword = passwordField.type === "password";

            passwordField.type = isPassword ? "text" : "password";

            toggleIcon.classList.toggle("fa-eye");
            toggleIcon.classList.toggle("fa-eye-slash");
        }
    });
}

function fetchSubcategories(categoryId, lang_id) {
    const subcategoriesContainer = document.getElementById("subcategories");

    if (!categoryId) {
        subcategoriesContainer.innerHTML = "";
        return;
    }

    subcategoriesContainer.innerHTML = `
        <div class="loader">Loading, please wait...</div>
    `;

    $.ajax({
        url: "/api/get-register-subcategories",
        type: "POST",
        data: { category_id: categoryId, language_code: lang_id },
        headers: {
            "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr("content"),
        },
        success: function (data) {
            let subcategoriesHtml = "";

            if (data.length > 0) {
                data.forEach((subcategory) => {
                    subcategoriesHtml += `
                        <div class="form-checkbox d-inline-flex align-items-center mb-2 me-3">
                            <input class="form-check-input ms-0 mt-0" 
                                name="subcategory_ids[]" 
                                type="checkbox" 
                                id="subcategory-${subcategory.id}" 
                                value="${subcategory.id}">
                            <label class="form-check-label ms-2" for="subcategory-${subcategory.id}">
                                ${subcategory.name}
                            </label>
                        </div>`;
                });
            } else {
                subcategoriesHtml = `
                    <div class="no-subcategories">No Subcategories Provided</div>
                `;
            }
            $("#subcategories").html(subcategoriesHtml);
            
            // Add change event listener to clear error when a subcategory is selected
            $(document).on('change', 'input[name="subcategory_ids[]"]', function() {
                if ($('input[name="subcategory_ids[]"]:checked').length > 0) {
                    $("#subcategory_ids_error").text("").css('display', 'none');
                }
            });
        },
        error: function (xhr) {
            const errorMessage =
                xhr.responseJSON && xhr.responseJSON.error
                    ? xhr.responseJSON.error
                    : "Failed to fetch subcategories. Please try again.";
            subcategoriesContainer.innerHTML = `
                <div class="error-message">${errorMessage}</div>
            `;
        },
    });
}

// Company/Individual selection removed - automatically set to individual
let subServiceType = "individual";

document.addEventListener("DOMContentLoaded", function () {
    // Load active languages for multi-select
    loadActiveLanguages();
    
    // Check if OTP modals exist before initializing
    const otpEmailModalEl = document.getElementById("otp-email-prov-reg-modal");
    const otpSmsModalEl = document.getElementById("otp-pro-reg-phone-modal");
    
    let otpEmailModal = null;
    let otpSmsModal = null;
    
    if (otpEmailModalEl && typeof bootstrap !== 'undefined' && bootstrap.Modal) {
        otpEmailModal = new bootstrap.Modal(otpEmailModalEl, { keyboard: false });
    }
    
    if (otpSmsModalEl && typeof bootstrap !== 'undefined' && bootstrap.Modal) {
        otpSmsModal = new bootstrap.Modal(otpSmsModalEl, { keyboard: false });
    }
    
    const otpTimerDisplay = document.getElementById("otp-pro-timer");
    const otpSmsTimerDisplay = document.getElementById("otp-pro-reg-sms-timer");
    const otpEmailMessage = document.querySelector("#otp-email-message");
    const successModel = document.querySelector("#register-modal");
    let timerInterval;
    let timerTime = 0;
    let userEmail = "";
    const otpModal = document.getElementById("otp-email-prov-reg-modal");
    const otpsmsModal = document.getElementById("otp-pro-reg-phone-modal");

    if (otpModal) {
        otpModal.addEventListener("hidden.bs.modal", function () {
        const errorMessage = document.getElementById(
            "error_prov_email_reg_message"
        );
        if (errorMessage) {
            errorMessage.textContent = "";
        }
        });
    }
    
    if (otpsmsModal) {
        otpsmsModal.addEventListener("hidden.bs.modal", function () {
        const errorMessage = document.getElementById(
            "error_pro_reg_sms_message"
        );
        if (errorMessage) {
            errorMessage.textContent = "";
        }
        });
    }

    function resendOtp(type) {
        const payload = {
            login_type: "register",
            ...userRegisterData,
        };

        const url = "/provider-register-otp-settings";

        $.ajax({
            url: url,
            type: "POST",
            data: payload,
            headers: {
                "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr("content"),
            },
            success: function (data) {
                const userName = data.name;
                const otp = data.otp;
                const otpExpireTime = parseInt(
                    data.otp_expire_time.split(" ")[0]
                );
                const phoneNumber = data.phone_number;
                // Get actual OTP type from response (in case settings changed)
                const actualOtpType = data.otp_type || type;

                showLoader();

                if (actualOtpType === "email") {
                    const emailData = {
                        subject: data.email_subject,
                        content: data.email_content,
                    };
                    // Correct parameter order: sendEmail(email, emailData, userName, otp)
                    sendEmail(data.email, emailData, userName, otp)
                        .then(() => {
                            hideLoader();
                            if (otpEmailModal) {
                                otpEmailModal.show();
                            } else if (otpEmailModalEl) {
                                $(otpEmailModalEl).modal('show');
                            }
                            startTimer(otpExpireTime);
                        })
                        .catch((error) => {
                            hideLoader();
                            $("#otp_error").modal("show");
                        });
                } else if (actualOtpType === "sms" || actualOtpType === "whatsapp") {
                    const emailData = {
                        subject: data.email_subject,
                        content: data.email_content,
                    };
                    // Correct parameter order: sendSms(phoneNumber, emailData, userName, otp, otpType)
                    sendSms(phoneNumber, emailData, userName, otp, actualOtpType)
                        .then(() => {
                            hideLoader();
                            const otpMethod = actualOtpType === "whatsapp" ? "WhatsApp" : "SMS";
                            // Update message to show WhatsApp or SMS
                            const otpSmsMessage = document.getElementById("otp-prov-reg-sms-message");
                            if (otpSmsMessage) {
                                otpSmsMessage.textContent = `OTP sent via ${otpMethod} to your mobile number ending ******${phoneNumber.slice(-4)}`;
                            }
                            otpSmsModal.show();
                            startSmsTimer(otpExpireTime);
                        })
                        .catch((error) => {
                            hideLoader();
                            const otpMethod = actualOtpType === "whatsapp" ? "WhatsApp" : "SMS";
                            alert(`Failed to send ${otpMethod} OTP. Please try again.`);
                        });
                } else {
                    hideLoader();
                }
            },
            error: function (xhr) {
                const errorMessage =
                    xhr.responseJSON && xhr.responseJSON.error
                        ? xhr.responseJSON.error
                        : "Failed to resend OTP. Please try again.";
                toastr.error(errorMessage);
                hideLoader();
            },
        });
    }

    // Provider registration resend OTP
    
    $(".resendProRegSMSOtp").on("click", function () {
        // Use the actual OTP type from settings instead of hardcoded "sms"
        resendOtp("whatsapp"); // This will be overridden by response data anyway
    });

    let emailTimerInterval, smsTimerInterval;
    let emailTimerTime, smsTimerTime;

    function startTimer(expireTime) {
        clearInterval(emailTimerInterval);
        emailTimerTime = expireTime * 60;

        emailTimerInterval = setInterval(() => {
            let minutes = Math.floor(emailTimerTime / 60);
            let seconds = emailTimerTime % 60;
            otpTimerDisplay.textContent = `${String(minutes).padStart(
                2,
                "0"
            )}:${String(seconds).padStart(2, "0")}`;
            emailTimerTime--;

            if (emailTimerTime < 0) {
                clearInterval(emailTimerInterval);
                otpTimerDisplay.textContent = "00:00"; // Timer finished
            }
        }, 1000);
    }

    function startSmsTimer(expireSmsTime) {
        clearInterval(smsTimerInterval); // Clear any existing timer
        smsTimerTime = expireSmsTime * 60; // Convert minutes to seconds

        smsTimerInterval = setInterval(() => {
            let minutes = Math.floor(smsTimerTime / 60);
            let seconds = smsTimerTime % 60;
            otpSmsTimerDisplay.textContent = `${String(minutes).padStart(
                2,
                "0"
            )}:${String(seconds).padStart(2, "0")}`;
            smsTimerTime--;

            if (smsTimerTime < 0) {
                clearInterval(smsTimerInterval);
                otpSmsTimerDisplay.textContent = "00:00";
            }
        }, 1000);
    }

    function resetTimer() {
        clearInterval(emailTimerInterval);
        otpTimerDisplay.textContent = "00:00";
    }

    function resetSmsTimer() {
        clearInterval(smsTimerInterval);
        otpSmsTimerDisplay.textContent = "00:00";
    }

    let userRegisterData = {};

    $("#provider_phone_number").on("input", function () {
        $(this).val(
            $(this)
                .val()
                .replace(/[^0-9]/g, "")
        );

        if ($(this).val().length > 12) {
            $(this).val($(this).val().slice(0, 12));
        }
    });

    // Bio character counter
    $("#provider_bio").on("input", function () {
        const length = $(this).val().length;
        const maxLength = 3000;
        const counter = $("#bio_char_count");
        
        counter.text(`${length} / ${maxLength}`);
        
        if (length > maxLength) {
            counter.removeClass("text-success text-muted").addClass("text-danger");
        } else {
            counter.removeClass("text-danger").addClass("text-muted");
        }
    });

    // Initialize intlTelInput only if provider phone input exists (might be on page or in modal)
    const phoneInput = document.querySelector("#provider_phone_number");
    let iti = null;
    if (phoneInput) {
        try {
            iti = intlTelInput(phoneInput, {
                utilsScript: "https://cdnjs.cloudflare.com/ajax/libs/intl-tel-input/17.0.8/js/utils.js",
                separateDialCode: true,
                initialCountry: "za",
                preferredCountries: ['za', 'us', 'gb'],
                autoPlaceholder: "off",
                placeholderNumberType: "MOBILE"
            });
        } catch (error) {
            console.warn("Failed to initialize intlTelInput for #provider_phone_number:", error);
        }
    }

    // Profile Picture Upload Functionality
    const profilePictureUpload = document.getElementById('profile-picture-upload');
    const profilePictureInput = document.getElementById('provider_profile_image');
    const profilePicturePreview = document.getElementById('profile-picture-preview');
    const profilePicturePlaceholder = document.getElementById('profile-picture-placeholder');
    const profilePreviewImg = document.getElementById('profile-preview-img');

    if (profilePictureUpload && profilePictureInput) {
        // Click to upload
        profilePictureUpload.addEventListener('click', function() {
            profilePictureInput.click();
        });

        // File input change
        profilePictureInput.addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                // Validate file type
                if (!file.type.startsWith('image/')) {
                    alert('Please select a valid image file (JPEG, PNG, JPG, GIF).');
                    profilePictureInput.value = '';
                    return;
                }
                
                // Validate file size (2MB max)
                if (file.size > 2 * 1024 * 1024) {
                    alert('File size must be less than 2MB.');
                    profilePictureInput.value = '';
                    return;
                }

                // Show preview
                const reader = new FileReader();
                reader.onload = function(e) {
                    profilePreviewImg.src = e.target.result;
                    profilePicturePreview.style.display = 'block';
                    profilePicturePlaceholder.style.display = 'none';
                };
                reader.readAsDataURL(file);
            }
        });

        // Drag and drop functionality
        profilePictureUpload.addEventListener('dragover', function(e) {
            e.preventDefault();
            profilePictureUpload.classList.add('dragover');
        });

        profilePictureUpload.addEventListener('dragleave', function(e) {
            e.preventDefault();
            profilePictureUpload.classList.remove('dragover');
        });

        profilePictureUpload.addEventListener('drop', function(e) {
            e.preventDefault();
            profilePictureUpload.classList.remove('dragover');
            
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                const file = files[0];
                if (file.type.startsWith('image/')) {
                    if (file.size <= 2 * 1024 * 1024) {
                        profilePictureInput.files = files;
                        const reader = new FileReader();
                        reader.onload = function(e) {
                            profilePreviewImg.src = e.target.result;
                            profilePicturePreview.style.display = 'block';
                            profilePicturePlaceholder.style.display = 'none';
                        };
                        reader.readAsDataURL(file);
                    } else {
                        alert('File size must be less than 2MB.');
                    }
                } else {
                    alert('Please select a valid image file (JPEG, PNG, JPG, GIF).');
                }
            }
        });
    }

    $(document).ready(function () {
        var langCode = $("body").data("lang");

        let currentF1Lang = langCode;
        let currentF2Lang = langCode;

        const validationF1Messages = {
            en: {
                provider_first_name: {
                    required: "The name field is required.",
                    minlength: "The name must be at least 1 character.",
                },
                provider_last_name: {
                    required: "The name field is required.",
                    minlength: "The name must be at least 1 character.",
                },
                provider_name: {
                    required: "The name field is required.",
                    minlength: "The name must be at least 1 character.",
                    remote: "Provider name already exist",
                },
                provider_email: {
                    required: "The email field is required.",
                    email: "Please enter a valid email address.",
                    pattern: "Please enter a valid email address.",
                    remote: "Given email already exist.",
                },
                provider_password: {
                    required: "The password field is required.",
                    minlength: "The password must be at least 8 characters.",
                },
                provider_phone_number: {
                    required: "The phone number field is required.",
                    minlength:
                        "The phone number must be at least 10 characters.",
                },
                provider_profile_image: {
                    required: "Profile picture is required.",
                },
                provider_dob: {
                    required: "Date of birth is required.",
                },
                provider_gender: {
                    required: "Gender is required.",
                },
                provider_bio: {
                    maxlength: "Bio cannot exceed 3000 characters.",
                },
                provider_address: {
                    required: "Address is required.",
                    maxlength: "Address cannot exceed 150 characters.",
                },
                provider_country: {
                    required: "Country is required.",
                },
                provider_state: {
                    required: "State is required.",
                },
                provider_city: {
                    required: "City is required.",
                },
                provider_postal_code: {
                    required: "Postal code is required.",
                    maxlength: "Postal code cannot exceed 10 characters.",
                },
                'provider_language[]': {
                    required: "At least one language is required.",
                },
            },
            ar: {
                provider_first_name: {
                    required: "حقل الاسم مطلوب.",
                    minlength: "يجب أن يكون الاسم حرف واحد على الأقل.",
                },
                provider_last_name: {
                    required: "حقل الاسم مطلوب.",
                    minlength: "يجب أن يكون الاسم حرف واحد على الأقل.",
                },
                provider_name: {
                    required: "حقل الاسم مطلوب.",
                    minlength: "يجب أن يكون الاسم حرف واحد على الأقل.",
                    remote: "اسم المزود موجود بالفعل.",
                },
                provider_email: {
                    required: "حقل البريد الإلكتروني مطلوب.",
                    email: "يرجى إدخال عنوان بريد إلكتروني صالح.",
                    pattern: "يرجى إدخال عنوان بريد إلكتروني صالح.",
                    remote: "البريد الإلكتروني المقدم موجود بالفعل.",
                },
                provider_password: {
                    required: "حقل كلمة المرور مطلوب.",
                    minlength: "يجب أن تكون كلمة المرور 8 أحرف على الأقل.",
                },
                provider_phone_number: {
                    required: "حقل رقم الهاتف مطلوب.",
                    minlength: "يجب أن يكون رقم الهاتف 10 أحرف على الأقل.",
                },
            },
        };

        const validationF2Messages = {
            en: {
                category_id: {
                    required: "The category field is required.",
                },
                'subcategory_ids[]': {
                    required: "Please select at least one subcategory.",
                },
                company_name: {
                    required: "The company name field is required.",
                    minlength:
                        "The company name must be at least 3 characters.",
                    maxlength:
                        "The company name must not exceed 50 characters.",
                },
                company_website: {
                    required: "The company website field is required.",
                    minlength:
                        "The company website must be at least 3 characters.",
                    maxlength:
                        "The company website must not exceed 255 characters.",
                    url: "Please enter a valid URL.",
                },
                provider_terms_policy: {
                    required:
                        "Please approve the Terms and Conditions & Privacy Policy.",
                },
            },
            ar: {
                category_id: {
                    required: "حقل الفئة مطلوب.",
                },
                company_name: {
                    required: "حقل اسم الشركة مطلوب.",
                    minlength: "يجب أن يكون اسم الشركة 3 أحرف على الأقل.",
                    maxlength: "يجب ألا يتجاوز اسم الشركة 50 حرفًا.",
                },
                company_website: {
                    required: "حقل موقع الشركة مطلوب.",
                    minlength: "يجب أن يكون موقع الشركة 3 أحرف على الأقل.",
                    maxlength: "يجب ألا يتجاوز موقع الشركة 255 حرفًا.",
                    url: "يرجى إدخال عنوان URL صالح.",
                },
                provider_terms_policy: {
                    required:
                        "يرجى الموافقة على الشروط والأحكام وسياسة الخصوصية.",
                },
            },
        };

        $("#providerRegister").validate({
            rules: {
                provider_first_name: {
                    required: true,
                    minlength: 1,
                    maxlength: 50,
                },
                provider_last_name: {
                    required: true,
                    minlength: 1,
                    maxlength: 50,
                },
                provider_name: {
                    required: true,
                    minlength: 1,
                    maxlength: 50,
                    remote: {
                        url: "/api/user/check-unique",
                        type: "post",
                        headers: {
                            Authorization:
                                "Bearer " + localStorage.getItem("admin_token"),
                            Accept: "application/json",
                        },
                        data: {
                            name: function () {
                                return $("#provider_name").val();
                            },
                            id: function () {
                                return $("#id").val();
                            },
                        },
                    },
                },
                provider_email: {
                    required: true,
                    email: true,
                    pattern: /^[a-zA-Z0-9._-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/,
                    remote: {
                        url: "/api/user/check-unique",
                        type: "post",
                        headers: {
                            Authorization:
                                "Bearer " + localStorage.getItem("admin_token"),
                            Accept: "application/json",
                        },
                        data: {
                            email: function () {
                                return $("#provider_email").val();
                            },
                            id: function () {
                                return $("#id").val();
                            },
                        },
                    },
                },
                provider_password: {
                    required: true,
                    minlength: 8,
                },
                provider_phone_number: {
                    required: true,
                    number: true,
                    minlength: 10,
                },
                provider_profile_image: {
                    required: true,
                },
                provider_dob: {
                    required: true,
                },
                provider_gender: {
                    required: true,
                },
                provider_bio: {
                    maxlength: 3000,
                },
                provider_address: {
                    required: true,
                    maxlength: 150,
                },
                provider_country: {
                    required: true,
                },
                provider_state: {
                    required: true,
                },
                provider_city: {
                    required: true,
                },
                provider_postal_code: {
                    required: true,
                    maxlength: 10,
                },
                'provider_language[]': {
                    required: true,
                },
            },
            messages: validationF1Messages[currentF1Lang],
            errorElement: "span",
            errorPlacement: function (error, element) {
                if (element.attr("name") === "category_id") {
                    error.appendTo("#category_id_error"); // Make sure you have an element with this id for error message.
                } else {
                    error.addClass("invalid-feedback");
                    element.closest(".mb-3").append(error); // Default error placement
                }
            },
            highlight: function (element, errorClass, validClass) {
                $(element).addClass("is-invalid").removeClass("is-valid");
            },
            unhighlight: function (element, errorClass, validClass) {
                $(element).removeClass("is-invalid").addClass("is-valid");
            },
        });

        $("#get_started_btn").on("click", function (event) {
            event.preventDefault();

            let infoFormData = $("#providerRegister").serializeArray();

            if ($("#providerRegister").valid()) {
                let formDataCollection = {};
                infoFormData.forEach(function (item) {
                    formDataCollection[item.name] = item.value;
                });

                $("#first-field").hide();
                $("#second-field").show();
            }
        });

        // Add custom validation method for subcategories checkboxes
        $.validator.addMethod("subcategoryRequired", function(value, element) {
            return $('input[name="subcategory_ids[]"]:checked').length > 0;
        }, "Please select at least one subcategory.");

        $("#companyInfo").validate({
            rules: {
                category_id: {
                    required: true,
                },
                company_name: {
                    required: false,
                    minlength: 3,
                    maxlength: 50,
                },
                company_website: {
                    required: false,
                    minlength: 3,
                    maxlength: 255,
                    url: true,
                },
                provider_terms_policy: {
                    required: true,
                },
            },
            messages: validationF2Messages[currentF2Lang],
            errorElement: "span",
            errorPlacement: function (error, element) {
                error.addClass("invalid-feedback");

                if (element.attr("name") === "provider_terms_policy") {
                    // Special case for provider_terms_policy
                    $("#provider_terms_policy_error").html(error);
                } else if (element.attr("name") === "category_id") {
                    // Special case for category_id
                    $("#category_id_error").html(error);
                } else {
                    // Append error to the closest parent div
                    element.closest(".mb-4").append(error);
                }
            },
            highlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element)
                        .next(".select2-container")
                        .addClass("is-invalid")
                        .removeClass("is-valid");
                } else {
                    $(element).addClass("is-invalid").removeClass("is-valid");
                }
            },
            unhighlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element)
                        .next(".select2-container")
                        .removeClass("is-invalid")
                        .addClass("is-valid");
                } else {
                    $(element).removeClass("is-invalid").addClass("is-valid");
                }
            },
        });

        $("#provider_register_btn").on("click", function (event) {
            event.preventDefault();

            let companyInfo = $("#companyInfo").serializeArray();
            let infoFormData = $("#providerRegister").serializeArray();

            // Custom validation for subcategories
            const subcategoriesChecked = $('input[name="subcategory_ids[]"]:checked').length > 0;
            if (!subcategoriesChecked) {
                $("#subcategory_ids_error").text("Please select at least one subcategory.").css('display', 'block');
                return false;
            } else {
                $("#subcategory_ids_error").text("").css('display', 'none');
            }

            if ($("#companyInfo").valid()) {
                let formDataCollection = {};
                companyInfo.forEach(function (item) {
                    formDataCollection[item.name] = item.value;
                });

                const all = 1;

                if (all == 1) {
                    let finalFormData = new FormData();

                    finalFormData.append(
                        "_token",
                        $('meta[name="csrf-token"]').attr("content")
                    );

                    [...infoFormData, ...companyInfo].forEach(function (item) {
                        finalFormData.append(item.name, item.value);
                    });

                    // Add profile picture if selected
                    const profileImageFile = document.getElementById('provider_profile_image').files[0];
                    if (profileImageFile) {
                        finalFormData.append('provider_profile_image', profileImageFile);
                        console.log('Profile image added:', profileImageFile.name, profileImageFile.size);
                    } else {
                        console.warn('No profile image selected!');
                    }

                    // Normalize phone number and location fields before submit
                    const fullPhoneNumber = iti ? iti.getNumber() : $("#provider_phone_number").val();
                    finalFormData.set("provider_phone_number", fullPhoneNumber);

                    // Ensure country/state/city are submitted as IDs (not names)
                    const selectedCountryId = $("#provider_country option:selected").data("id") || $("#provider_country").val();
                    const selectedStateId = $("#provider_state option:selected").data("id") || $("#provider_state").val();
                    const selectedCityId = $("#provider_city option:selected").data("id") || $("#provider_city").val();

                    if (selectedCountryId) {
                        finalFormData.set("provider_country", String(selectedCountryId));
                    }
                    if (selectedStateId) {
                        finalFormData.set("provider_state", String(selectedStateId));
                    }
                    if (selectedCityId) {
                        finalFormData.set("provider_city", String(selectedCityId));
                    }

                    finalFormData.append("sub_service_type", subServiceType);

                    // Manually add checked subcategories to FormData
                    const checkedSubcategories = [];
                    $('input[name="subcategory_ids[]"]:checked').each(function() {
                        checkedSubcategories.push($(this).val());
                        finalFormData.append('subcategory_ids[]', $(this).val());
                    });
                    
                    // Debug log
                    console.log('Checked subcategories:', checkedSubcategories);

                    $.ajax({
                        url: "/provider/register",
                        method: "POST",
                        data: finalFormData,
                        dataType: "json",
                        contentType: false,
                        processData: false,
                        cache: false,
                        headers: {
                            "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr(
                                "content"
                            ),
                        },
                        beforeSend: function () {
                            $(".provider_register_btn").attr("disabled", true);
                            $(".provider_register_btn").html(
                                '<div class="spinner-border text-light" role="status"></div>'
                            );
                        },
                    })
                        .done((response, statusText, xhr) => {
                            $(".error-text").text("");
                            $(".form-control").removeClass("is-invalid");
                            $(".provider_register_btn").removeAttr("disabled");
                            $(".provider_register_btn").html("Submit");

                            if (response.code === 200) {
                                if (response.register_status === "0") {
                                    $("#provider").modal("hide");

                                    if (response.provider_approval_status == 1) {
                                        $("#provider_approval_success_modal").modal("show");
                                    } else {
                                        $("#reg_success_modal").modal("show");

                                        setTimeout(function () {
                                            location.reload();
                                        }, 500);
                                    }

                                } else if (response.register_status === "1") {
                                    $("#provider").modal("hide");

                                    userRegisterData = {
                                        name: response.name,
                                        phone_number: response.phone_number,
                                        email: response.email,
                                        password: response.password,
                                        category_id: response.category_id,
                                        subcategory_ids:
                                            response.subcategory_ids,
                                        company_name: response.company_name,
                                        company_website:
                                            response.company_website,
                                        provider_first_name:
                                            response.provider_first_name,
                                        provider_last_name:
                                            response.provider_last_name,
                                        // Include DOB, gender, and address fields
                                        dob: response.dob,
                                        gender: response.gender,
                                        bio: response.bio,
                                        address: response.address,
                                        city: response.city,
                                        state: response.state,
                                        country: response.country,
                                        postal_code: response.postal_code,
                                        language: response.language,
                                        profile_image: response.profile_image
                                    };

                                    const userName = response.name;
                                    const otp = response.otp;
                                    const otpExpireTime = parseInt(
                                        response.otp_expire_time.split(" ")[0]
                                    );
                                    const phoneNumber = response.phone_number;

                                    showLoader();

                                    const otpDigitLimit = parseInt(
                                        response.otp_digit_limit
                                    );

                                    const inputProvideContainerreg = $(
                                        ".inputProvideContainerreg"
                                    );
                                    inputProvideContainerreg.empty();

                                    let inputsHtml =
                                        '<div class="d-flex align-items-center mb-3">';
                                    for (let i = 1; i <= otpDigitLimit; i++) {
                                        const nextId = `digit-${i + 1}`;
                                        const prevId = `digit-${i - 1}`;
                                        inputsHtml += `
                                <input type="text"
                                       class="rounded w-100 py-sm-3 py-2 text-center fs-26 fw-bold me-3 digit-${i}"
                                       id="digit-${i}"
                                       name="digit-${i}"
                                       data-next="${nextId}"
                                       data-previous="${prevId}"
                                       maxlength="1">
                            `;
                                    }

                                    inputsHtml += "</div>";
                                    inputProvideContainerreg.append(inputsHtml);

                                    $(".inputProvideContainerreg").on(
                                        "input",
                                        "input",
                                        function () {
                                            const maxLength =
                                                $(this).attr("maxlength") || 1;
                                            if (
                                                this.value.length >= maxLength
                                            ) {
                                                const next =
                                                    $(this).data("next");
                                                if (next) {
                                                    $("#" + next).focus();
                                                }
                                            }
                                        }
                                    );

                                    $(".inputProvideContainerreg").on(
                                        "keydown",
                                        "input",
                                        function (e) {
                                            if (
                                                e.key === "Backspace" &&
                                                this.value === ""
                                            ) {
                                                const prev =
                                                    $(this).data("previous");
                                                if (prev) {
                                                    $("#" + prev).focus();
                                                }
                                            }
                                        }
                                    );

                                    $(".inputProvideContainerreg").on(
                                        "click",
                                        "input",
                                        function () {
                                            $(this).select();
                                        }
                                    );

                                    const inputProRegSMSContainer = $(
                                        ".inputProRegSMSContainer"
                                    );
                                    inputProRegSMSContainer.empty(); // Clear any existing input fields

                                    let inputsSMSHtml =
                                        '<div class="d-flex align-items-center mb-3">';
                                    for (let i = 1; i <= otpDigitLimit; i++) {
                                        const nextId = `digitsms-${i + 1}`;
                                        const prevId = `digitsms-${i - 1}`;
                                        inputsSMSHtml += `
                                <input type="text"
                                       class="rounded w-100 py-sm-3 py-2 text-center fs-26 fw-bold me-3 digitsms-${i}"
                                       id="digitsms-${i}"
                                       name="digitsms-${i}"
                                       data-next="${nextId}"
                                       data-previous="${prevId}"
                                       maxlength="1">
                            `;
                                    }

                                    inputsSMSHtml += "</div>";
                                    inputProRegSMSContainer.append(
                                        inputsSMSHtml
                                    );

                                    $(".inputProRegSMSContainer").on(
                                        "input",
                                        "input",
                                        function () {
                                            const maxLength =
                                                $(this).attr("maxlength") || 1;
                                            if (
                                                this.value.length >= maxLength
                                            ) {
                                                const next =
                                                    $(this).data("next");
                                                if (next) {
                                                    $("#" + next).focus();
                                                }
                                            }
                                        }
                                    );

                                    $(".inputProRegSMSContainer").on(
                                        "keydown",
                                        "input",
                                        function (e) {
                                            if (
                                                e.key === "Backspace" &&
                                                this.value === ""
                                            ) {
                                                const prev =
                                                    $(this).data("previous");
                                                if (prev) {
                                                    $("#" + prev).focus();
                                                }
                                            }
                                        }
                                    );

                                    $(".inputProRegSMSContainer").on(
                                        "click",
                                        "input",
                                        function () {
                                            $(this).select();
                                        }
                                    );

                                    if (response.otp_type === "email") {
                                        const emailData = {
                                            subject: response.email_subject,
                                            content: response.email_content,
                                        };
                                        sendEmail(
                                            userName,
                                            emailData,
                                            "email",
                                            userName,
                                            otp
                                        )
                                            .then(() => {
                                                hideLoader();
                                                const otpEmailMessage =
                                                    document.getElementById(
                                                        "otp-prov-reg-email-message"
                                                    );
                                                if (otpEmailMessage) {
                                                    otpEmailMessage.textContent = `OTP sent to your Email Address ${userName}`;
                                                }
                                                otpEmailModal.show();
                                                startTimer(otpExpireTime);
                                            })
                                            .catch((error) => {
                                                hideLoader();
                                                $("#otp_error").modal("show");
                                            });
                                    } else if (response.otp_type === "sms" || response.otp_type === "whatsapp") {
                                        const emailData = {
                                            subject: response.email_subject,
                                            content: response.email_content,
                                        };
                                        sendSms(
                                            phoneNumber,
                                            emailData,
                                            userName,
                                            otp,
                                            response.otp_type
                                        )
                                            .then(() => {
                                                hideLoader();
                                                const otpSmsMessage =
                                                    document.getElementById(
                                                        "otp-prov-reg-sms-message"
                                                    );
                                                if (otpSmsMessage) {
                                                    const messageText = response.otp_type === "whatsapp" 
                                                        ? `OTP sent to your WhatsApp number ending ******${phoneNumber.slice(-4)}`
                                                        : `OTP sent to your mobile number ending ******${phoneNumber.slice(-4)}`;
                                                    otpSmsMessage.textContent = messageText;
                                                }
                                                otpSmsModal.show();
                                                startSmsTimer(otpExpireTime);
                                            })
                                            .catch((error) => {
                                                hideLoader();
                                                $("#otp_error").modal("show");
                                            });
                                    } else {
                                        hideLoader();
                                    }
                                }
                            } else {
                                alert(
                                    "Registration completed, but OTP setup failed."
                                );
                            }
                        })

                        .fail((xhr, statusText, error) => {
                            $(".error-text").text(""); // Clear all previous error messages
                            $(".form-control").removeClass("is-invalid"); // Remove invalid classes
                            $(".provider_register_btn").removeAttr("disabled"); // Re-enable the button
                            $(".provider_register_btn").html("Submit");

                            console.error('Registration failed:', xhr);
                            console.error('Response:', xhr.responseText);

                            if (xhr.status === 422) {
                                // Validation error
                                let response = JSON.parse(xhr.responseText);
                                console.error('Validation errors:', response);
                                
                                // Display validation errors
                                if (response.errors) {
                                    $.each(response.errors, function(field, messages) {
                                        console.error('Field:', field, 'Errors:', messages);
                                        toastr.error(messages[0]);
                                    });
                                } else {
                                    toastr.error(
                                        response.message ||
                                            "An error occurred. Please try again."
                                    );
                                }
                            } else if (xhr.status === 500) {
                                console.error('Server error 500:', xhr.responseText);
                                toastr.error("Server error. Please check the console for details.");
                            } else {
                                toastr.error("An unexpected error occurred.");
                            }
                        });
                }
            }
        });
    });

    $("#verify-email-prov-reg-otp-btn").on("click", function () {
        const otpDigitLimit = $(".inputProvideContainerreg input").length;

        // Collect the entered OTP digits
        const otp = [];
        for (let i = 1; i <= otpDigitLimit; i++) {
            const digit = $(`#digit-${i}`).val();
            otp.push(digit);
        }
        const otpString = otp.join("");

        // Add userRegisterData to the payload
        const payload = {
            otp: otpString,
            login_type: "provider_register",
            ...userRegisterData, // Include name, phone_number, email, password
        };

        $.ajax({
            url: "/verify-otp",
            type: "POST",
            data: payload,
            headers: {
                "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr("content"),
            },
            beforeSend: function () {
                $(".verify-email-prov-reg-otp-btn").attr("disabled", true);
                $(".verify-email-prov-reg-otp-btn").html(
                    '<div class="spinner-border text-light" role="status"></div>'
                );
            },
            success: function (response) {
                $("#otp-email-prov-reg-modal").modal("hide");

                $("#reg_success_modal").modal("show");

                setTimeout(function () {
                    // Check if ID verification is required
                    if (response.next_step === 'id_verification') {
                        // Redirect to ID verification page
                        window.location.href = "/provider/identity-verification";
                    } else if (response.provider_approval_status == 1) {
                        // Provider needs approval - redirect to home with message
                        window.location.href = "/";
                    } else {
                        // Provider approved - redirect to dashboard
                        window.location.href = "/provider/dashboard";
                    }
                }, 500);
            },
            error: function (xhr) {
                const errorMessage = xhr.responseJSON.error || "OTP Required";

                $("#error_prov_email_reg_message").text(errorMessage);
            },
            complete: function () {
                $(".verify-email-prov-reg-otp-btn").attr("disabled", false);
                $(".verify-email-prov-reg-otp-btn").html("Verify OTP");
            },
        });
    });

    $("#verify-pro-reg-sms-otp-btn").on("click", function () {
        const otpDigitLimit = $(".inputProRegSMSContainer input").length;

        const otp = [];
        for (let i = 1; i <= otpDigitLimit; i++) {
            const digit = $(`#digitsms-${i}`).val();
            otp.push(digit);
        }
        const otpString = otp.join("");

        const payload = {
            otp: otpString,
            login_type: "provider_register",
            ...userRegisterData,
        };

        $.ajax({
            url: "/verify-otp",
            type: "POST",
            data: payload,
            headers: {
                "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr("content"),
            },
            beforeSend: function () {
                $(".verify-pro-reg-sms-otp-btn").attr("disabled", true);
                $(".verify-pro-reg-sms-otp-btn").html(
                    '<div class="spinner-border text-light" role="status"></div>'
                );
            },
            success: function (response) {
                $("#otp-pro-reg-phone-modal").modal("hide");

                $("#reg_success_modal").modal("show");

                setTimeout(function () {
                    // Check if ID verification is required
                    if (response.next_step === 'id_verification') {
                        // Redirect to ID verification page
                        window.location.href = "/provider/identity-verification";
                    } else if (response.provider_approval_status == 1) {
                        // Provider needs approval - redirect to home with message
                        window.location.href = "/";
                    } else {
                        // Provider approved - redirect to dashboard
                        window.location.href = "/provider/dashboard";
                    }
                }, 500);
            },
            error: function (xhr) {
                const errorMessage = xhr.responseJSON.error || "OTP Required";
                $("#error_pro_reg_sms_message").text(errorMessage);
            },
            complete: function () {
                $(".verify-pro-reg-sms-otp-btn").attr("disabled", false);
                $(".verify-pro-reg-sms-otp-btn").html("Verify OTP");
            },
        });
    });

    function isValidEmail(email) {
        const emailPattern = /^[a-zA-Z0-9._-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/;
        return emailPattern.test(email);
    }

    function sendEmail(email, emailData, userName, otp) {
        return new Promise((resolve, reject) => {
            $.ajax({
                url: "/api/mail/sendmail",
                type: "POST",
                dataType: "json",
                data: {
                    otp_type: "email",
                    to_email: email,
                    notification_type: 2,
                    type: 1,
                    user_name: userName,
                    otp: otp,
                    subject: emailData.subject,
                    content: emailData.content,
                },
                headers: {
                    Authorization:
                        "Bearer " + localStorage.getItem("admin_token"),
                    Accept: "application/json",
                },
                success: function (response) {
                    resolve(response);
                },
                error: function (error) {
                    reject(error);
                },
            });
        });
    }

    function sendSms(phoneNumber, emailData, userName, otp, otpType) {
        return new Promise((resolve, reject) => {
            // Default to 'sms' if not provided (for backwards compatibility)
            otpType = otpType || 'sms';
            $.ajax({
                url: "/api/sms/sendsms",
                type: "POST",
                dataType: "json",
                data: {
                    otp_type: otpType,
                    to_number: phoneNumber,
                    notification_type: 2,
                    type: 1,
                    user_name: userName,
                    otp: otp,
                    subject: emailData.subject,
                    content: emailData.content,
                },
                headers: {
                    Authorization:
                        "Bearer " + localStorage.getItem("admin_token"),
                    Accept: "application/json",
                },
                success: function (response) {
                    resolve(response);
                },
                error: function (error) {
                    if (error.status === 422) {
                        const errors = error.responseJSON.errors;
                        if (errors) {
                        } else {
                            toastr.error(
                                "An error occurred while sending OTP."
                            );
                        }
                    } else {
                        toastr.error("An error occurred while sending OTP.");
                    }
                    reject(error);
                },
            });
        });
    }

    function showLoader() {
        const loader = document.getElementById("pageLoader");
        if (loader) {
            loader.style.display = "block";
        }
    }

    function hideLoader() {
        const loader = document.getElementById("pageLoader");
        if (loader) {
            loader.style.display = "none";
        }
    }

    // if (loginForm && emailInput) {
    //     loginForm.addEventListener("submit", function (event) {
    //         userEmail = emailInput.value;
    //     });
    // }
});

document.addEventListener("DOMContentLoaded", function () {
    // Select the modal element (only if it exists - might not be on registration page)
    const providerModal = document.getElementById("provider");

    // Attach the 'hidden.bs.modal' event listener only if modal exists
    if (providerModal) {
        providerModal.addEventListener("hidden.bs.modal", function () {
        // Reset the form inside the modal
        const form = providerModal.querySelector("#providerRegister");
        if (form) {
            form.reset(); // Reset all form inputs to their initial state
        }

        // Clear custom error messages if any
        providerModal
            .querySelectorAll(".invalid-feedback")
            .forEach(function (errorElement) {
                errorElement.textContent = ""; // Clear error messages
            });

        // Hide dynamically added elements (e.g., company details)
        const companyDetails = providerModal.querySelector("#company_details");
        if (companyDetails) {
            companyDetails.style.display = "none"; // Hide the company details section
        }

        // Reset dynamically populated subcategories
        const subcategories = providerModal.querySelector("#subcategories");
        if (subcategories) {
            subcategories.innerHTML = ""; // Clear subcategory checkboxes
        }

        // Reset dropdown selections
        const categorySelect = providerModal.querySelector("#categorySelect");
        if (categorySelect) {
            categorySelect.value = ""; // Set the default value
        }
        });
    }
});

// Provider Registration - Country, State, City Dropdowns
$(document).ready(function () {
    let cachedCountries = [];
    let cachedStates = [];
    let cachedCities = [];
    let locationDataLoaded = false;

    // Load location data when modal is shown
    $('#provider').on('shown.bs.modal', function () {
        if (!locationDataLoaded) {
            loadLocationData();
        }
    });

    async function loadLocationData() {
        try {
            const [countriesResponse, statesResponse, citiesResponse] = await Promise.all([
                $.getJSON("/countries.json"),
                $.getJSON("/states.json"),
                $.getJSON("/cities.json"),
            ]);

            cachedCountries = countriesResponse.countries;
            cachedStates = statesResponse.states;
            cachedCities = citiesResponse.cities;

            locationDataLoaded = true;
            getCountries();
        } catch (error) {
            console.error("Failed to load location data:", error);
        }
    }

    function clearDropdown(dropdown) {
        dropdown.empty().append(
            $("<option>", {
                value: "",
                text: "Select",
                disabled: true,
                selected: true,
            })
        );
    }

    function getCountries(selectedCountry = null) {
        const countrySelect = $("#provider_country");
        clearDropdown(countrySelect);

        $.each(cachedCountries, function (index, country) {
            countrySelect.append(
                $("<option>", {
                    value: country.name,
                    text: country.name,
                    'data-id': country.id,
                    selected: country.name == selectedCountry,
                })
            );
        });

        if (selectedCountry) {
            const selectedCountryId = cachedCountries.find(c => c.name == selectedCountry)?.id;
            if (selectedCountryId) {
                getStates(selectedCountryId);
            }
        }
    }

    function getStates(selectedCountryId, selectedState = null) {
        const stateSelect = $("#provider_state");
        clearDropdown(stateSelect);

        const states = cachedStates.filter(
            (state) => state.country_id == selectedCountryId
        );

        // Sort states alphabetically by name
        states.sort((a, b) => a.name.localeCompare(b.name));

        if (states.length === 1) {
            stateSelect.append(
                $("<option>", {
                    value: states[0].name,
                    text: states[0].name,
                    'data-id': states[0].id,
                    selected: true,
                })
            );
            getCities(states[0].id);
        } else {
            $.each(states, function (index, state) {
                stateSelect.append(
                    $("<option>", {
                        value: state.name,
                        text: state.name,
                        'data-id': state.id,
                        selected: state.name == selectedState,
                    })
                );
            });

            if (selectedState) {
                const selectedStateId = states.find(s => s.name == selectedState)?.id;
                if (selectedStateId) {
                    getCities(selectedStateId);
                }
            }
        }
    }

    function getCities(selectedStateId, selectedCity = null) {
        const citySelect = $("#provider_city");
        clearDropdown(citySelect);

        const cities = cachedCities.filter(
            (city) => city.state_id == selectedStateId
        );

        // Sort cities alphabetically by name
        cities.sort((a, b) => a.name.localeCompare(b.name));

        $.each(cities, function (index, city) {
            citySelect.append(
                $("<option>", {
                    value: city.name,
                    text: city.name,
                    'data-id': city.id,
                    selected: city.name == selectedCity,
                })
            );
        });
    }

    // Event listeners for provider registration form
    $("#provider_country").on("change", function () {
        const selectedCountryName = $(this).val();
        const selectedCountryId = $(this).find('option:selected').data('id');
        
        clearDropdown($("#provider_state"));
        clearDropdown($("#provider_city"));
        
        if (selectedCountryId) {
            getStates(selectedCountryId);
        }
    });

    $("#provider_state").on("change", function () {
        const selectedStateId = $(this).find('option:selected').data('id');
        
        clearDropdown($("#provider_city"));
        
        if (selectedStateId) {
            getCities(selectedStateId);
        }
    });
});
