<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>{{ $template->name }} - Form Preview</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/remixicon@2.5.0/fonts/remixicon.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            min-height: 100vh;
        }
        .preview-container {
            max-width: 800px;
            margin: 0 auto;
            padding: 2rem;
        }
        .form-card {
            background: white;
            border-radius: 12px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        .form-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem;
            text-align: center;
            border-radius: 12px 12px 0 0;
            box-shadow: 0 4px 20px rgba(102, 126, 234, 0.3);
        }
        
        .form-header h1 {
            font-weight: 700;
            font-size: 1.75rem;
            margin-bottom: 0.5rem;
        }
        
        .form-header p {
            opacity: 0.9;
            font-size: 1rem;
            margin: 0;
        }
        .form-body {
            padding: 2rem;
        }
        .section-title {
            color: #2d3748;
            font-weight: 700;
            margin-bottom: 1.5rem;
            padding-bottom: 0.75rem;
            border-bottom: 3px solid #667eea;
            font-size: 1.25rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        .question-item {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 1.5rem;
            margin-bottom: 1rem;
        }
        .question-text {
            font-weight: 500;
            margin-bottom: 1rem;
            color: #333;
        }
        .options-container {
            display: flex;
            gap: 1rem;
        }
        .option-item {
            flex: 1;
            text-align: center;
        }
        .option-label {
            display: block;
            padding: 0.75rem 1rem;
            background: white;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.2s ease;
            font-weight: 500;
        }
        .option-label:hover {
            border-color: #667eea;
            background: #f8f9ff;
        }
        .option-input:checked + .option-label {
            border-color: #667eea;
            background: #667eea;
            color: white;
        }
        .option-input {
            display: none;
        }
        
        /* Ranking Interface Styles */
        .ranking-instructions {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 1rem;
        }
        
        .ranking-options {
            display: flex;
            flex-direction: column;
            gap: 0.75rem;
        }
        
        .ranking-option {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            background: white;
            cursor: pointer;
            transition: all 0.2s ease;
            position: relative;
        }
        
        .ranking-option:hover {
            border-color: #667eea;
            background: #f8f9ff;
        }
        
        .ranking-option.ranked {
            border-color: #28a745;
            background: #f8fff9;
        }
        
        .ranking-option.ranked-1 {
            border-color: #ffc107;
            background: #fffdf5;
        }
        
        .ranking-option.ranked-2 {
            border-color: #17a2b8;
            background: #f5fdff;
        }
        
        .ranking-option.ranked-3 {
            border-color: #6c757d;
            background: #f8f9fa;
        }
        
        .ranking-number {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            font-size: 14px;
            color: white;
            background: #6c757d;
            transition: all 0.2s ease;
        }
        
        .ranking-number.rank-1 {
            background: #ffc107;
        }
        
        .ranking-number.rank-2 {
            background: #17a2b8;
        }
        
        .ranking-number.rank-3 {
            background: #6c757d;
        }
        
        .ranking-label {
            flex: 1;
            font-weight: 500;
        }
        .personality-result {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            padding: 2rem;
            border-radius: 12px;
            text-align: center;
            margin-top: 2rem;
            display: none;
        }
        .personality-type {
            font-size: 2rem;
            font-weight: bold;
            margin-bottom: 0.5rem;
        }
        .personality-description {
            font-size: 1.1rem;
            opacity: 0.9;
        }
        .progress-bar {
            height: 8px;
            background: #e9ecef;
            border-radius: 4px;
            overflow: hidden;
            margin-bottom: 1rem;
        }
        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #667eea 0%, #764ba2 100%);
            width: 0%;
            transition: width 0.3s ease;
        }
        .close-btn {
            position: fixed;
            top: 1rem;
            right: 1rem;
            background: #dc3545;
            color: white;
            border: none;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            z-index: 1000;
        }
    </style>
</head>
<body>
    <!-- Close button removed for preview -->

    <div class="preview-container">
        <!-- Original Form Preview -->
        <div class="form-card">
            <div class="form-header">
                <h1>{{ $template->name }}</h1>
                <p class="mb-0">Paradigm Profile Assessment</p>
            </div>

            <div class="form-body">
                <form id="paradigm-assessment-form">
                    @csrf
                    <input type="hidden" name="template_id" value="{{ $template->id }}">

                    <!-- Personal Information Section -->
                    @if($template->personalFields->count() > 0)
                    <div class="mb-4">
                        <h4 class="section-title">
                            <i class="ri-user-line me-2"></i>Personal Information
                        </h4>
                        
                        <div class="row">
                            @foreach($template->personalFields->sortBy('display_order') as $field)
                                @php
                                    // Skip practitioner-specific fields
                                    $practitionerFields = [
                                        'practitioner_name',
                                        'practitioner_email', 
                                        'practitioner_phone',
                                        'practitioner_website'
                                    ];
                                    
                                    $isPractitionerField = in_array($field->field_key, $practitionerFields) || 
                                                          str_contains(strtolower($field->field_label), 'practitioner');
                                @endphp
                                
                                @if(!$isPractitionerField)
                                <div class="col-md-6 mb-3">
                                    <label for="field_{{ $field->id }}" class="form-label">
                                        {{ $field->field_label }}
                                        @if($field->is_required)
                                            <span class="text-danger">*</span>
                                        @endif
                                    </label>
                                    
                                    @if($field->field_type === 'select')
                                        <select class="form-select" id="field_{{ $field->id }}" name="personal[{{ $field->field_key }}]" {{ $field->is_required ? 'required' : '' }}>
                                            <option value="">{{ __('Select an option') }}</option>
                                            @if($field->options)
                                                @if(is_array($field->options))
                                                    @foreach($field->options as $option)
                                                        <option value="{{ $option }}">{{ $option }}</option>
                                                    @endforeach
                                                @else
                                                    @foreach(explode("\n", $field->options) as $option)
                                                        <option value="{{ trim($option) }}">{{ trim($option) }}</option>
                                                    @endforeach
                                                @endif
                                            @endif
                                        </select>
                                    @else
                                        <input type="{{ $field->field_type }}" 
                                               class="form-control" 
                                               id="field_{{ $field->id }}" 
                                               name="personal[{{ $field->field_key }}]"
                                               {{ $field->is_required ? 'required' : '' }}
                                               placeholder="{{ __('Enter') }} {{ strtolower($field->field_label) }}">
                                    @endif
                                </div>
                                @endif
                            @endforeach
                        </div>
                    </div>
                    @endif

                    <!-- Assessment Questions Section -->
                    @if($template->questions->count() > 0)
                    <div class="mb-4">
                        <h4 class="section-title">
                            <i class="ri-question-line me-2"></i>Assessment Questions
                        </h4>
                        
                        <div class="progress-bar">
                            <div class="progress-fill" id="progress-fill"></div>
                        </div>
                        <p class="text-muted text-center">
                            <span id="current-question">1</span> of {{ $template->questions->count() }} questions
                        </p>

                        @foreach($template->questions->sortBy('display_order') as $index => $question)
                        <div class="question-item" data-question="{{ $index + 1 }}" style="{{ $index > 0 ? 'display: none;' : '' }}">
                            <div class="question-text">
                                {{ $question->question_text }}
                            </div>
                            
                            <div class="options-container">
                                @if($question->assessment_module === 'brain_frames')
                                    <!-- Ranking Interface for Brain Profile 4 Frame -->
                                    <div class="ranking-instructions mb-3">
                                        <p class="text-muted small">
                                            <strong>Instructions:</strong> Choose your top 3 preferences by clicking in order of preference.
                                        </p>
                                    </div>
                                    
                                    <div class="ranking-options">
                                        @if($question->option_a_label)
                                        <div class="ranking-option" data-option="A" data-question="{{ $question->display_order }}">
                                            <div class="ranking-number" id="rank_{{ $question->display_order }}_a"></div>
                                            <div class="ranking-label">{{ $question->option_a_label }}</div>
                                        </div>
                                        @endif
                                        
                                        @if($question->option_b_label)
                                        <div class="ranking-option" data-option="B" data-question="{{ $question->display_order }}">
                                            <div class="ranking-number" id="rank_{{ $question->display_order }}_b"></div>
                                            <div class="ranking-label">{{ $question->option_b_label }}</div>
                                        </div>
                                        @endif
                                        
                                        @if($question->option_c_label)
                                        <div class="ranking-option" data-option="C" data-question="{{ $question->display_order }}">
                                            <div class="ranking-number" id="rank_{{ $question->display_order }}_c"></div>
                                            <div class="ranking-label">{{ $question->option_c_label }}</div>
                                        </div>
                                        @endif
                                        
                                        @if($question->option_d_label)
                                        <div class="ranking-option" data-option="D" data-question="{{ $question->display_order }}">
                                            <div class="ranking-number" id="rank_{{ $question->display_order }}_d"></div>
                                            <div class="ranking-label">{{ $question->option_d_label }}</div>
                                        </div>
                                        @endif
                                    </div>
                                    
                                    <!-- Hidden inputs to store ranking -->
                                    <input type="hidden" name="questions[{{ $question->display_order }}_A]" id="hidden_{{ $question->display_order }}_A" value="">
                                    <input type="hidden" name="questions[{{ $question->display_order }}_B]" id="hidden_{{ $question->display_order }}_B" value="">
                                    <input type="hidden" name="questions[{{ $question->display_order }}_C]" id="hidden_{{ $question->display_order }}_C" value="">
                                    <input type="hidden" name="questions[{{ $question->display_order }}_D]" id="hidden_{{ $question->display_order }}_D" value="">
                                    
                                @else
                                    <!-- Regular Radio Button Interface for other questions -->
                                    <div class="option-item">
                                        <input type="radio" 
                                               class="option-input" 
                                               id="q{{ $question->id }}_a" 
                                               name="questions[{{ $question->display_order }}]" 
                                               value="A"
                                               onchange="nextQuestion()">
                                        <label for="q{{ $question->id }}_a" class="option-label">
                                            {{ $question->option_a_label }}
                                        </label>
                                    </div>
                                    
                                    <div class="option-item">
                                        <input type="radio" 
                                               class="option-input" 
                                               id="q{{ $question->id }}_b" 
                                               name="questions[{{ $question->display_order }}]" 
                                               value="B"
                                               onchange="nextQuestion()">
                                        <label for="q{{ $question->id }}_b" class="option-label">
                                            {{ $question->option_b_label }}
                                        </label>
                                    </div>
                                @endif
                            </div>
                        </div>
                        @endforeach
                    </div>
                    @endif

                    <!-- Submit Button -->
                    <div class="text-center">
                        <button type="submit" class="btn btn-primary btn-lg" id="submit-btn" style="display: none;">
                            <i class="ri-check-line me-2"></i>Complete Assessment
                        </button>
                    </div>
                </form>

                <!-- Results Section -->
                <div class="personality-result" id="personality-result">
                    <div class="personality-type" id="personality-type"></div>
                    <div class="personality-description" id="personality-description"></div>
                </div>
            </div>
        </div>
    </div>

    <script>
        console.log('Paradigm Preview Script Loading - Version 1.0');
        let currentQuestion = 1;
        const totalQuestions = {{ $template->questions->count() }};
        const questions = document.querySelectorAll('.question-item');
        
        // Question mapping data from database - using question ID as key
        const questionMapping = @json($template->questions->mapWithKeys(function($question) {
            return [$question->display_order => [
                'A' => $question->a_side,
                'B' => $question->b_side
            ]];
        }));
        
        console.log('Question mapping loaded:', questionMapping);
        console.log('Question IDs in mapping:', Object.keys(questionMapping));
        console.log('Type of questionMapping:', typeof questionMapping);
        console.log('Is questionMapping an object:', questionMapping instanceof Object);
        console.log('Question mapping keys count:', Object.keys(questionMapping).length);

        function updateProgress() {
            const progress = (currentQuestion / totalQuestions) * 100;
            document.getElementById('progress-fill').style.width = progress + '%';
            document.getElementById('current-question').textContent = currentQuestion;
        }

        function nextQuestion() {
            // Hide current question
            document.querySelector(`[data-question="${currentQuestion}"]`).style.display = 'none';
            
            currentQuestion++;
            
            if (currentQuestion <= totalQuestions) {
                // Show next question
                document.querySelector(`[data-question="${currentQuestion}"]`).style.display = 'block';
                updateProgress();
            } else {
                // All questions answered, show submit button
                document.getElementById('submit-btn').style.display = 'block';
                // Don't update progress beyond total questions
                currentQuestion = totalQuestions;
                updateProgress();
            }
        }

        // Ranking functionality for Brain Profile 4 Frame questions
        function initializeRanking() {
            // Add click handlers to all ranking options
            document.querySelectorAll('.ranking-option').forEach(option => {
                option.addEventListener('click', function() {
                    const questionNum = this.getAttribute('data-question');
                    const optionKey = this.getAttribute('data-option');
                    handleRankingClick(questionNum, optionKey, this);
                });
            });
        }

        function handleRankingClick(questionNum, optionKey, element) {
            const rankingContainer = element.closest('.ranking-options');
            const currentRankings = getCurrentRankings(questionNum);
            
            // If already ranked, remove the ranking
            if (currentRankings[optionKey]) {
                removeRanking(questionNum, optionKey, element);
                return;
            }
            
            // If we already have 3 rankings, don't allow more
            if (Object.keys(currentRankings).length >= 3) {
                alert('You can only select your top 3 preferences. Click on a ranked option to remove it first.');
                return;
            }
            
            // Add the ranking
            const nextRank = Object.keys(currentRankings).length + 1;
            addRanking(questionNum, optionKey, nextRank, element);
        }

        function getCurrentRankings(questionNum) {
            const rankings = {};
            document.querySelectorAll(`[data-question="${questionNum}"] .ranking-option`).forEach(option => {
                const optionKey = option.getAttribute('data-option');
                const rankNumber = option.querySelector('.ranking-number');
                if (rankNumber.textContent) {
                    rankings[optionKey] = parseInt(rankNumber.textContent);
                }
            });
            return rankings;
        }

        function addRanking(questionNum, optionKey, rank, element) {
            const rankNumber = element.querySelector('.ranking-number');
            const hiddenInput = document.getElementById(`hidden_${questionNum}_${optionKey}`);
            
            // Set the ranking
            rankNumber.textContent = rank;
            rankNumber.className = `ranking-number rank-${rank}`;
            element.className = `ranking-option ranked ranked-${rank}`;
            
            // Set the hidden input value (format: "1st", "2nd", etc.)
            const rankText = rank === 1 ? '1st' : rank === 2 ? '2nd' : '3rd';
            hiddenInput.value = rankText;
            
            // Check if we have 3 rankings, then auto-advance
            const currentRankings = getCurrentRankings(questionNum);
            if (Object.keys(currentRankings).length >= 3) {
                setTimeout(() => {
                    nextQuestion();
                }, 500);
            }
        }

        function removeRanking(questionNum, optionKey, element) {
            const rankNumber = element.querySelector('.ranking-number');
            const hiddenInput = document.getElementById(`hidden_${questionNum}_${optionKey}`);
            const removedRank = parseInt(rankNumber.textContent);
            
            // Clear this ranking
            rankNumber.textContent = '';
            rankNumber.className = 'ranking-number';
            element.className = 'ranking-option';
            hiddenInput.value = '';
            
            // Adjust other rankings
            document.querySelectorAll(`[data-question="${questionNum}"] .ranking-option`).forEach(option => {
                const otherOptionKey = option.getAttribute('data-option');
                const otherRankNumber = option.querySelector('.ranking-number');
                const otherHiddenInput = document.getElementById(`hidden_${questionNum}_${otherOptionKey}`);
                
                if (otherRankNumber.textContent) {
                    const otherRank = parseInt(otherRankNumber.textContent);
                    if (otherRank > removedRank) {
                        const newRank = otherRank - 1;
                        otherRankNumber.textContent = newRank;
                        otherRankNumber.className = `ranking-number rank-${newRank}`;
                        option.className = `ranking-option ranked ranked-${newRank}`;
                        
                        const rankText = newRank === 1 ? '1st' : newRank === 2 ? '2nd' : '3rd';
                        otherHiddenInput.value = `${otherOptionKey}${rankText}`;
                    }
                }
            });
        }

        // Initialize ranking when page loads
        document.addEventListener('DOMContentLoaded', function() {
            initializeRanking();
        });

        // Paradigm Scoring Functions
        function calculateParadigmScores(answers) {
            // Use the question mapping data from the database
            
            // Initialize trait counts for Paradigm modules
            const traitCounts = {
                // Balance Profile
                'balance_profile': 0,
                // Conflict Styles
                'conflict_styles': 0,
                // Connection Styles
                'connection_styles': 0,
                // Personality Blueprint
                'personality_blueprint': 0,
                // SERQ
                'serq': 0,
                // Brain Frames
                'brain_frames': 0,
                // Brain Fields
                'brain_fields': 0,
                // Type Classification
                'type_classification': 0
            };
            
            // Process each answer according to the mapping
            for (const [questionKey, answer] of Object.entries(answers)) {
                const questionId = questionKey.replace('questions[', '').replace(']', '');
                console.log(`Looking for question ID: ${questionId} (type: ${typeof questionId})`);
                console.log(`Mapping exists: ${questionMapping.hasOwnProperty(questionId)}`);
                
                if (!questionMapping[questionId]) {
                    console.log(`No mapping found for question ${questionId}`);
                    continue;
                }
                
                const mapping = questionMapping[questionId];
                const trait = answer === 'A' ? mapping.A : mapping.B;
                console.log(`Q${questionId}: ${answer} -> ${trait}`);
                
                if (traitCounts.hasOwnProperty(trait)) {
                    traitCounts[trait]++;
                }
            }
            
            console.log('Trait counts:', traitCounts);
            
            // Calculate percentages for each trait pair
            const scores = {};
            
            // Calculate module scores
            Object.keys(traitCounts).forEach(module => {
                scores[module] = traitCounts[module];
            });
            
            console.log('Final scores:', scores);
            console.log('=== END CALCULATION ===');
            
            return scores;
        }
        
        function calculateParadigmType(scores) {
            // Paradigm uses a different classification system
            // This is a simplified version - you can expand based on your specific logic
            let type = 'Paradigm Profile';
            
            // Example logic - adjust based on your Paradigm classification system
            if (scores.balance_profile > 0) {
                type += ' - Balanced';
            }
            if (scores.conflict_styles > 0) {
                type += ' - Conflict Resolver';
            }
            if (scores.connection_styles > 0) {
                type += ' - Connector';
            }
            
            return type;
        }
        
        function getParadigmDescription(typeCode) {
            return 'You have completed the Paradigm Profile assessment. This comprehensive evaluation provides insights into your personality, conflict resolution style, connection patterns, and more.';
        }

        // PDF Download and Refresh Functions
        function downloadPDF() {
            // Get actual form data instead of hardcoded sample data
            const formData = new FormData(document.getElementById('paradigm-assessment-form'));
            const personalData = {};
            const answers = {};
            
            // Extract personal data
            for (let [key, value] of formData.entries()) {
                if (key.startsWith('personal[')) {
                    const field = key.replace('personal[', '').replace(']', '');
                    personalData[field] = value;
                } else if (key.startsWith('questions[')) {
                    answers[key] = value;
                }
            }
            
            // Calculate actual scores using Paradigm formula
            const scores = calculateParadigmScores(answers);
            const typeCode = calculateParadigmType(scores);
            
            // Create assessment data with actual form responses
            const sampleData = {
                personal_data: {
                    name: personalData.name || 'John',
                    surname: personalData.surname || 'Doe',
                    email: personalData.email || 'john.doe@example.com',
                    gender: personalData.gender || 'Male',
                    age: personalData.age || '25-34',
                    phone: personalData.phone || '+1 (555) 123-4567'
                },
                scores: scores,
                special: {
                    type: typeCode,
                    description: getParadigmDescription(typeCode)
                },
                result: {
                    completed_at: new Date().toISOString(),
                    session_id: 'SES-123456789'
                }
            };
            
            // Submit form to generate PDF
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = '/admin/paradigm/templates/{{ $template->id }}/generate-pdf-preview';
            form.target = '_blank';
            
            // Add CSRF token
            const csrfInput = document.createElement('input');
            csrfInput.type = 'hidden';
            csrfInput.name = '_token';
            csrfInput.value = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
            form.appendChild(csrfInput);
            
            // Add sample data
            Object.keys(sampleData).forEach(key => {
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = key;
                input.value = JSON.stringify(sampleData[key]);
                form.appendChild(input);
            });
            
            document.body.appendChild(form);
            form.submit();
            document.body.removeChild(form);
        }
        
        function refreshPreview() {
            window.location.reload();
        }

        // Form submission
        document.getElementById('paradigm-assessment-form').addEventListener('submit', function(e) {
            e.preventDefault();
            
            // Get form data
            const formData = new FormData(this);
            const personalData = {};
            const answers = {};
            
            // Extract personal data and answers
            for (let [key, value] of formData.entries()) {
                if (key.startsWith('personal[')) {
                    const field = key.replace('personal[', '').replace(']', '');
                    personalData[field] = value;
                } else if (key.startsWith('questions[')) {
                    answers[key] = value;
                }
            }
            
            // Calculate scores using proper Paradigm formula
            const scores = calculateParadigmScores(answers);
            const type = calculateParadigmType(scores);
            
            // Show loading state
            const submitBtn = document.getElementById('submit-btn');
            if (!submitBtn) {
                console.error('Submit button not found');
                alert('Error: Submit button not found. Please refresh the page and try again.');
                return;
            }
            
            const originalText = submitBtn.innerHTML;
            submitBtn.innerHTML = '<i class="ri-loader-4-line ri-spin"></i> Saving & Generating PDF...';
            submitBtn.disabled = true;
            
            // First, save results to database (same as frontend form)
            const saveFormData = new FormData();
            saveFormData.append('template_id', '{{ $template->id }}');
            saveFormData.append('personal', JSON.stringify(personalData));
            saveFormData.append('questions', JSON.stringify(answers));
            
            // Get CSRF token
            const csrfToken = document.querySelector('meta[name="csrf-token"]');
            if (!csrfToken) {
                console.error('CSRF token not found');
                alert('Error: CSRF token not found. Please refresh the page and try again.');
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
                return;
            }
            
            // Save to database first
            fetch('/paradigm/assessment/submit', {
                method: 'POST',
                body: saveFormData,
                headers: {
                    'X-CSRF-TOKEN': csrfToken.getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    console.log('Assessment saved to database successfully');
                    
                    // Now generate PDF (same as before)
                    const assessmentData = {
                        personal_data: {
                            name: personalData.name || 'John',
                            surname: personalData.surname || 'Doe',
                            email: personalData.email || 'john.doe@example.com',
                            gender: personalData.gender || 'Male',
                            age: personalData.age || '25-34',
                            phone: personalData.phone || '+1 (555) 123-4567'
                        },
                        scores: scores,
                        special: {
                            type: type,
                            description: getParadigmDescription(type)
                        },
                        result: {
                            completed_at: new Date().toISOString(),
                            session_id: 'SES-' + Math.random().toString(36).substr(2, 9)
                        }
                    };
                    
                    // Submit to PDF generation endpoint
                    const pdfForm = document.createElement('form');
                    pdfForm.method = 'POST';
                    pdfForm.action = '/admin/paradigm/templates/{{ $template->id }}/generate-pdf-preview';
                    pdfForm.target = '_blank';
                    
                    // Add CSRF token
                    const csrfInput = document.createElement('input');
                    csrfInput.type = 'hidden';
                    csrfInput.name = '_token';
                    csrfInput.value = csrfToken.getAttribute('content');
                    pdfForm.appendChild(csrfInput);
                    
                    // Add assessment data
                    Object.keys(assessmentData).forEach(key => {
                        const input = document.createElement('input');
                        input.type = 'hidden';
                        input.name = key;
                        input.value = JSON.stringify(assessmentData[key]);
                        pdfForm.appendChild(input);
                    });
                    
                    document.body.appendChild(pdfForm);
                    pdfForm.submit();
                    document.body.removeChild(pdfForm);
                    
                    // Show success message
                    submitBtn.innerHTML = '<i class="ri-check-line"></i> Complete! Check Admin Results';
                    
                } else {
                    throw new Error(data.message || 'Failed to save assessment');
                }
            })
            .catch(error => {
                console.error('Error saving assessment:', error);
                alert('Error saving assessment: ' + error.message + '. PDF will still be generated.');
                
                // Fallback: Generate PDF even if database save fails
                const assessmentData = {
                    personal_data: {
                        name: personalData.name || 'John',
                        surname: personalData.surname || 'Doe',
                        email: personalData.email || 'john.doe@example.com',
                        gender: personalData.gender || 'Male',
                        age: personalData.age || '25-34',
                        phone: personalData.phone || '+1 (555) 123-4567'
                    },
                    scores: scores,
                    special: {
                        type: type,
                        description: getParadigmDescription(type)
                    },
                    result: {
                        completed_at: new Date().toISOString(),
                        session_id: 'SES-' + Math.random().toString(36).substr(2, 9)
                    }
                };
                
                // Submit to PDF generation endpoint
                const pdfForm = document.createElement('form');
                pdfForm.method = 'POST';
                pdfForm.action = '/admin/paradigm/templates/{{ $template->id }}/generate-pdf-preview';
                pdfForm.target = '_blank';
                
                // Add CSRF token
                const csrfInput = document.createElement('input');
                csrfInput.type = 'hidden';
                csrfInput.name = '_token';
                csrfInput.value = csrfToken.getAttribute('content');
                pdfForm.appendChild(csrfInput);
                
                // Add assessment data
                Object.keys(assessmentData).forEach(key => {
                    const input = document.createElement('input');
                    input.type = 'hidden';
                    input.name = key;
                    input.value = JSON.stringify(assessmentData[key]);
                    pdfForm.appendChild(input);
                });
                
                document.body.appendChild(pdfForm);
                pdfForm.submit();
                document.body.removeChild(pdfForm);
                
                submitBtn.innerHTML = '<i class="ri-file-download-line"></i> PDF Generated';
            })
            .finally(() => {
                // Reset button after delay
                setTimeout(() => {
                    submitBtn.innerHTML = originalText;
                    submitBtn.disabled = false;
                }, 3000);
            });
        });

        // Initialize
        updateProgress();
    </script>
</body>
</html>
