@extends('layouts.clean')

@section('title', $template->name . ' - PARADIGM PROFILES™ Assessment')
{{-- Updated: Part 4 Conflict changes applied --}}

@section('styles')
    <style>
        /* Preloader Styles */
        #paradigm-preloader {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: #ffffff;
            z-index: 9999;
            display: flex;
            justify-content: center;
            align-items: center;
            transition: opacity 0.5s ease-out;
        }
        
        .spinner {
            width: 50px;
            height: 50px;
            border: 5px solid #f3f3f3;
            border-top: 5px solid #044f54;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* Mobile Responsiveness Improvements */
        @media (max-width: 768px) {
            .paradigm-assessment-form {
                padding: 0 10px;
            }
            
            .personal-details-card {
                padding: 1.5rem !important;
            }
            
            .paradigm-question {
                padding: 1rem;
            }
            
            .ranking-option, .radio-option {
                padding: 1rem; /* Larger touch target */
                margin-bottom: 0.75rem;
            }
            
            .btn {
                width: 100%; /* Full width buttons on mobile */
                margin-bottom: 0.5rem;
            }
            
            .d-flex.justify-content-between {
                flex-direction: column-reverse;
                gap: 10px;
            }
            
            #paradigm-next-btn, #paradigm-submit-btn {
                width: 100%;
            }
            
            #paradigm-prev-btn {
                width: 100%;
            }
        }

        .paradigm-assessment-form {
            max-width: 800px;
            margin: 0 auto;
        }
        
        .paradigm-form-page {
            min-height: 400px;
        }
        
        .paradigm-question {
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 1.5rem;
            background: #f8f9fa;
            margin-bottom: 1rem;
        }
        
        .paradigm-question-text {
            color: #333;
            margin-bottom: 1rem;
            font-weight: 500;
        }
        
        .paradigm-question-options .form-check {
            margin-bottom: 0.5rem;
        }
        
        .paradigm-progress-container {
            background: #f8f9fa;
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 2rem;
        }
        
        .paradigm-navigation {
            background: #f8f9fa;
            padding: 1rem;
            border-radius: 8px;
        }
        
        .module-section {
            background-color: #044f54;
            color: white;
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1rem;
        }
        
        .module-title {
            font-weight: bold;
            margin-bottom: 0.5rem;
        }
        
        .module-description {
            font-size: 0.9rem;
            opacity: 0.9;
        }
    .ranking-option, .radio-option {
        padding: 0.75rem 1.25rem;
        border: 1px solid #dee2e6;
        border-radius: 0.25rem;
        margin-bottom: 0.5rem;
        cursor: pointer;
        transition: all 0.2s ease-in-out;
        position: relative;
        background-color: #fff;
    }
    .ranking-option:hover, .radio-option:hover {
        background-color: #f8f9fa;
        border-color: #044f54;
    }
    .ranking-option.selected, .radio-option.selected {
        background-color: #044f54;
        border-color: #044f54;
        color: white;
    }
    .ranking-badge, .radio-badge {
        position: absolute;
        top: 50%;
        left: -15px;
        transform: translateY(-50%);
        background-color: #044f54 !important;
        color: white !important;
        width: 30px;
        height: 30px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: bold;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
    /* Modern Form Styling */
    .personal-details-card {
        max-width: 800px;
        margin: 1rem auto;
        padding: 2.5rem;
        background-color: #ffffff;
        border-radius: 15px;
        box-shadow: 0 8px 24px rgba(0, 40, 100, 0.08);
        border: 1px solid #e9ecef;
    }

    .personal-details-card .page-title {
        color: #044f54; /* Custom Teal Color */
        font-weight: 600;
        margin-bottom: 2rem !important;
    }

    .form-label {
        font-weight: 500;
        color: #495057;
    }

    .form-control, .form-select {
        border-radius: 8px;
        padding: 0.75rem 1rem;
        border-color: #ced4da;
    }
    .form-control:focus, .form-select:focus {
        border-color: #044f54;
        box-shadow: 0 0 0 0.25rem rgba(4, 79, 84, 0.25);
    }

    /* Ensure intl-tel-input takes full width inside the grid */
    .iti {
        width: 100%;
    }

    /* Auto-populated field styling */
    .form-control[readonly], .form-select[disabled] {
        background-color: #f8f9fa;
        border-color: #e9ecef;
        color: #6c757d;
        cursor: not-allowed;
    }

    .form-control[readonly]:focus, .form-select[disabled]:focus {
        box-shadow: none;
        border-color: #e9ecef;
    }

    /* Form header styling */
    .paradigm-form-subtitle {
        font-size: 1.1rem;
        font-weight: 500;
        color: #6c757d;
        margin-bottom: 1rem;
    }

    /* Part 7 Balance radio buttons - ensure vertical stacking on all screen sizes */
    .balance-options-group {
        display: flex;
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .balance-options-group .form-check {
        margin-bottom: 0.5rem;
    }
    
    /* Part 2-7 Radio button styling - Selected state */
    .form-check-input:checked {
        background-color: #044f54 !important;
        border-color: #044f54 !important;
    }
    
    .form-check-input:focus {
        border-color: #044f54 !important;
        box-shadow: 0 0 0 0.25rem rgba(4, 79, 84, 0.25) !important;
    }
    
    
    /* Terms and Conditions link color */
    a[data-bs-target="#termsModal"] {
        color: #65a0bf !important;
    }
    </style>
@endsection

@section('content')
    <!-- Preloader -->
    <div id="paradigm-preloader">
        <div class="spinner"></div>
    </div>

    <div class="container py-5">
        <div class="paradigm-assessment-form">
            <!-- Form Header -->
            <div class="paradigm-form-header text-center mb-4">
                <div class="alert alert-primary text-white" role="alert" style="background-color: #65a0bf; border-color: #65a0bf;">
                    <strong>The PARADIGM PROFILES™ is the only instrument on the planet that reveals how you think, connect, lead, and react, mapping your mind, relationships, and leadership in one powerful profile.</strong>
                </div>
            </div>
            
            <!-- Progress Bar -->
            <div class="paradigm-progress-container">
                <div class="progress" style="height: 30px;">
                    <div class="progress-bar" role="progressbar" style="width: 0%; background-color: #044f54;" id="paradigm-progress-bar">
                        <span id="paradigm-progress-text" style="font-weight: bold; font-size: 14px; color: white;">0%</span>
                    </div>
                </div>
                <!-- Page indicator removed as per requirements -->
            </div>
            
            <!-- Form Container -->
            <form id="paradigm-assessment-form" class="paradigm-form" data-template-id="{{ $template->id }}">
                @csrf
                <input type="hidden" name="template_id" value="{{ $template->id }}">
                @if(isset($access_token))
                    <input type="hidden" name="access_token" value="{{ $access_token }}">
                @endif
                
                <!-- Personal Details Page -->
                <div class="paradigm-form-page active" data-page="1">
                    <div class="personal-details-card">
                        <h5 class="page-title text-center">Personal Details</h5>
                        <fieldset>
                            <legend class="visually-hidden">Personal Details</legend>
                            @foreach ($personal_fields->chunk(2) as $chunk)
                                <div class="row gx-4">
                                    @foreach ($chunk as $field)
                                        <div class="col-md-6">
                                            <div class="mb-4">
                                                <label for="personal_{{ $field->field_key }}" class="form-label">{{ $field->field_label }} @if($field->is_required)<span class="text-danger">*</span>@endif</label>
                                                @if ($field->field_type === 'select')
                                                    @php
                                                        // Get the correct field value using the same mapping logic
                                                        $selectFieldValue = '';
                                                        if ($user_profile) {
                                                            
                                                            switch($field->field_key) {
                                                                case 'age':
                                                                case 'age_range':
                                                                case 'age_group':
                                                                    // Try age_group first, then calculate from dob
                                                                    if ($user_profile->age_group) {
                                                                        $selectFieldValue = $user_profile->age_group;
                                                                    } elseif ($user_profile->dob) {
                                                                        $age = \Carbon\Carbon::parse($user_profile->dob)->age;
                                                                        
                                                                        if ($age < 18) $selectFieldValue = 'Under 18';
                                                                        elseif ($age >= 18 && $age <= 24) $selectFieldValue = '18-24';
                                                                        elseif ($age >= 25 && $age <= 34) $selectFieldValue = '25-34';
                                                                        elseif ($age >= 35 && $age <= 44) $selectFieldValue = '35-44';
                                                                        elseif ($age >= 45 && $age <= 54) $selectFieldValue = '45-54';
                                                                        elseif ($age >= 55 && $age <= 64) $selectFieldValue = '55-64';
                                                                        else $selectFieldValue = '65 or over';
                                                                    } else {
                                                                        $selectFieldValue = '';
                                                                    }
                                                                    break;
                                                                case 'gender':
                                                                    $genderValue = $user_profile->gender ?? '';
                                                                    // Convert lowercase to capitalized for form compatibility
                                                                    $selectFieldValue = ucfirst($genderValue);
                                                                    break;
                                                                default:
                                                                    $selectFieldValue = $user_profile->{$field->field_key} ?? '';
                                                            }
                                                        }
                                                    @endphp
                                                    <select class="form-select" id="personal_{{ $field->field_key }}" name="personal[{{ $field->field_key }}]" @if($field->is_required) required @endif @if($user_profile) readonly disabled @endif>
                                                        <option value="" disabled @if(!$selectFieldValue) selected @endif>Select an option</option>
                                                        @foreach ($field->getOptionsArray() as $option)
                                                            <option value="{{ $option }}" @if($selectFieldValue == $option) selected @endif>{{ $option }}</option>
                                                        @endforeach
                                                    </select>
                                                @elseif ($field->field_type === 'tel')
                                                    <input type="tel" class="form-control" id="personal_{{ $field->field_key }}" name="personal[{{ $field->field_key }}_visible]" value="{{ $user_profile ? ($user_profile->phone_number ?? $user_profile->mobile_number ?? $user_profile->tel ?? $user_profile->telephone ?? (auth()->check() ? auth()->user()->phone_number : '')) : '' }}" @if($field->is_required) required @endif @if($user_profile) readonly @endif>
                                                    <input type="hidden" name="personal[{{ $field->field_key }}]" value="{{ $user_profile ? ($user_profile->phone_number ?? $user_profile->mobile_number ?? $user_profile->tel ?? $user_profile->telephone ?? (auth()->check() ? auth()->user()->phone_number : '')) : '' }}">
                                                @else
                                                    @php
                                                        $fieldValue = '';
                                                        if ($user_profile) {
                                                            switch($field->field_key) {
                                                                case 'name':
                                                                    $fieldValue = $user_profile->first_name ?? '';
                                                                    break;
                                                                case 'surname':
                                                                    $fieldValue = $user_profile->last_name ?? '';
                                                                    break;
                                                                case 'email':
                                                                    $fieldValue = $user_profile->email ?? (auth()->check() ? auth()->user()->email : '');
                                                                    break;
                                                                case 'age':
                                                                case 'age_range':
                                                                case 'age_group':
                                                                    // Try age_group first, then calculate from dob
                                                                    if ($user_profile->age_group) {
                                                                        $fieldValue = $user_profile->age_group;
                                                                    } elseif ($user_profile->dob) {
                                                                        $age = \Carbon\Carbon::parse($user_profile->dob)->age;
                                                                        
                                                                        if ($age < 18) $fieldValue = 'Under 18';
                                                                        elseif ($age >= 18 && $age <= 24) $fieldValue = '18-24';
                                                                        elseif ($age >= 25 && $age <= 34) $fieldValue = '25-34';
                                                                        elseif ($age >= 35 && $age <= 44) $fieldValue = '35-44';
                                                                        elseif ($age >= 45 && $age <= 54) $fieldValue = '45-54';
                                                                        elseif ($age >= 55 && $age <= 64) $fieldValue = '55-64';
                                                                        else $fieldValue = '65 or over';
                                                                    } else {
                                                                        $fieldValue = '';
                                                                    }
                                                                    break;
                                                                case 'gender':
                                                                    $genderValue = $user_profile->gender ?? '';
                                                                    // Convert lowercase to capitalized for form compatibility
                                                                    $fieldValue = ucfirst($genderValue);
                                                                    break;
                                                                case 'phone':
                                                                case 'mobile':
                                                                case 'mobile_number':
                                                                case 'phone_number':
                                                                case 'tel':
                                                                case 'telephone':
                                                                    // Check both userDetails and main user table for phone number
                                                                    $fieldValue = $user_profile->phone_number ?? $user_profile->mobile_number ?? $user_profile->tel ?? $user_profile->telephone ?? (auth()->check() ? auth()->user()->phone_number : '') ?? '';
                                                                    break;
                                                                default:
                                                                    $fieldValue = $user_profile->{$field->field_key} ?? '';
                                                            }
                                                        }
                                                    @endphp
                                                    <input type="{{ $field->field_type }}" class="form-control" id="personal_{{ $field->field_key }}" name="personal[{{ $field->field_key }}]" value="{{ $fieldValue }}" @if($field->is_required) required @endif @if($user_profile) readonly @endif>
                                                @endif
                                            </div>
                                        </div>
                                    @endforeach
                                </div>
                            @endforeach
                        </fieldset>
                        
                        <!-- Terms and Conditions Section -->
                        <div class="row mt-4">
                            <div class="col-12">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="terms_conditions" name="terms_conditions" required>
                                    <label class="form-check-label" for="terms_conditions">
                                        I accept the <a href="#" class="text-primary" data-bs-toggle="modal" data-bs-target="#termsModal">Terms and Conditions</a>
                                    </label>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Start Button -->
                        <div class="row mt-4">
                            <div class="col-12 text-center">
                                <button type="button" class="btn btn-lg" id="paradigm-start-btn" style="background-color: #044f54; color: white; border-color: #044f54;">
                                    <i class="ri-play-line me-2"></i>Start Assessment
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Questions Pages -->
                @php
                $questionsPerPage = 1;
                // Group by 'axis' and sort by custom part order (1-7)
                $partOrder = [
                    'paradigm_main_part1' => 1,  // Part 1
                    'part2' => 2,                 // Part 2
                    'E_I' => 3,                   // Part 3
                    'conflict' => 4,              // Part 4
                    'connection' => 5,            // Part 5
                    'serq_si' => 6,               // Part 6 - SI
                    'serq_eq' => 7,               // Part 6 - EQ
                    'serq_rq' => 8,               // Part 6 - RQ
                    'balance_emotional' => 9,     // Part 7 - Emotional
                    'balance_spiritual' => 10,    // Part 7 - Spiritual
                    'balance_physical' => 11,     // Part 7 - Physical
                    'balance_social' => 12,       // Part 7 - Social
                    'balance_financial' => 13,    // Part 7 - Financial
                    'balance_occupational' => 14, // Part 7 - Occupational
                    'balance_intellectual' => 15, // Part 7 - Intellectual
                    'balance_environmental' => 16 // Part 7 - Environmental
                ];
                
                $questions = $template->questions->groupBy('axis')->sortBy(function($group, $axis) use ($partOrder) {
                    return $partOrder[$axis] ?? 999; // Unknown axes go to end
                });
                $pageOffset = 1; // Start pages after personal details
                @endphp
                
                @foreach($questions as $module => $moduleQuestions)
                    @php
                        $modulePages = $moduleQuestions->chunk($questionsPerPage);
                    @endphp
                    
                    @foreach($modulePages as $pageNumber => $pageQuestions)
                    <div class="paradigm-form-page" data-page="{{ $pageOffset }}" style="display: none;">
                            <div class="module-section">
                                <div class="module-title">
                                    @if($module === 'paradigm_main_part1')
                                        PARADIGM PROFILES™ Instructions
                                    @elseif($module === 'part2')
                                        INSTRUCTIONS: Select the option that resonates most with you from the following statements
                                    @elseif($module === 'E_I')
                                        INSTRUCTIONS: Select the option that resonates most with you from the following statements
                                    @elseif($module === 'conflict')
                                        INSTRUCTIONS: Select the option that resonates most with you from the following statements
                                    @elseif($module === 'connection')
                                        INSTRUCTIONS: Select the option that resonates most with you from the following statements
                                    @elseif($module === 'serq_si' || $module === 'serq_eq' || $module === 'serq_rq')
                                        INSTRUCTIONS: Select the option that resonates most with you from the following statements
                                    @elseif($module === 'balance_emotional' || $module === 'balance_spiritual' || $module === 'balance_physical' || $module === 'balance_social' || $module === 'balance_financial' || $module === 'balance_occupational' || $module === 'balance_intellectual' || $module === 'balance_environmental')
                                        INSTRUCTIONS: For each statement below, rate how often it applies to you using the following scale
                                    @else
                                        {{ ucwords(str_replace('_', ' ', $module)) }}
                                    @endif
                                </div>
                                <div class="module-description">
                                @if($module === 'paradigm_main_part1')
                                    For each question, choose your top 3 options by clicking in order of preference:
                                    <ul style="font-size: 0.8rem; margin-top: 0.5rem;">
                                        <li>1st click – Most likely</li>
                                        <li>2nd click – Second most likely</li>
                                        <li>3rd click – Third most likely</li>
                                    </ul>
                                    REMEMBER - There are no correct or incorrect answers.
                                @elseif($module === 'paradigm_main_part2')
                                    Select the option that resonates most with you from the following statements.
                                @else
                                    @switch($module)
                                        @case('balance_profile')
                                            Assess your overall life balance across 8 key dimensions
                                            @break
                                        @case('conflict_styles')
                                            Discover your approach to handling conflicts and disagreements
                                            @break
                                        @case('connection_styles')
                                            Understand how you prefer to connect with others
                                            @break
                                        @case('personality_blueprint')
                                            Explore your extroversion and introversion preferences
                                            @break
                                        @case('serq')
                                            Measure your Social, Emotional, and Relevancy Quotients
                                            @break
                                        @case('brain_frames')
                                            Identify your cognitive processing preferences
                                            @break
                                        @case('brain_fields')
                                            Discover your thinking and working style preferences
                                            @break
                                        @case('type_classification')
                                            Determine your overall PARADIGM PROFILES™ type
                                            @break
                                    @endswitch
                                @endif
                                </div>
                            </div>
                            
                            <div class="card">
                                @if($module !== 'paradigm_main_part1' && $module !== 'part2' && $module !== 'E_I' && $module !== 'conflict' && $module !== 'connection' && $module !== 'serq_si' && $module !== 'serq_eq' && $module !== 'serq_rq' && $module !== 'balance_emotional' && $module !== 'balance_spiritual' && $module !== 'balance_physical' && $module !== 'balance_social' && $module !== 'balance_financial' && $module !== 'balance_occupational' && $module !== 'balance_intellectual' && $module !== 'balance_environmental')
                                <div class="card-header">
                                    <h5 class="mb-0">{{ ucwords(str_replace('_', ' ', $module)) }} - Questions {{ $pageNumber + 1 }} of {{ $modulePages->count() }}</h5>
                                </div>
                                @endif
                                <div class="card-body">
                                    @foreach($pageQuestions as $questionIndex => $question)
                                    <div class="paradigm-question" id="question-{{ $question->id }}">
                                            <h6 class="paradigm-question-text">
                                                @if($module === 'part2')
                                                    Question {{ $pageNumber + 1 }}
                                                @elseif($module === 'conflict')
                                                    Question {{ $question->display_order }}
                                                @else
                                                    {{ $question->question_text }}
                                                @endif
                                            </h6>
                                            <div class="paradigm-question-options">

                                            {{-- Part 7: Rating Scale Questions (1-4) - CHECK FIRST! --}}
                                                @if($question->assessment_module === 'paradigm_balance')
                                                    <div class="balance-options-group">
                                                        <div class="form-check">
                                                            <input class="form-check-input balance-radio" 
                                                                   type="radio" 
                                                                   name="questions[{{ $question->id }}]" 
                                                                   id="q{{ $question->id }}_1" 
                                                                   value="1"
                                                                   data-question-id="{{ $question->id }}"
                                                                   required>
                                                            <label class="form-check-label" for="q{{ $question->id }}_1">
                                                                1 - Rarely, if ever
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input balance-radio" 
                                                                   type="radio" 
                                                                   name="questions[{{ $question->id }}]" 
                                                                   id="q{{ $question->id }}_2" 
                                                                   value="2"
                                                                   data-question-id="{{ $question->id }}"
                                                                   required>
                                                            <label class="form-check-label" for="q{{ $question->id }}_2">
                                                                2 - Sometimes
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input balance-radio" 
                                                                   type="radio" 
                                                                   name="questions[{{ $question->id }}]" 
                                                                   id="q{{ $question->id }}_3" 
                                                                   value="3"
                                                                   data-question-id="{{ $question->id }}"
                                                                   required>
                                                            <label class="form-check-label" for="q{{ $question->id }}_3">
                                                                3 - Most of the time
                                                            </label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input balance-radio" 
                                                                   type="radio" 
                                                                   name="questions[{{ $question->id }}]" 
                                                                   id="q{{ $question->id }}_4" 
                                                                   value="4"
                                                                   data-question-id="{{ $question->id }}"
                                                                   required>
                                                            <label class="form-check-label" for="q{{ $question->id }}_4">
                                                                4 - Always
                                                            </label>
                                                        </div>
                                                    </div>

                                            {{-- Part 1: Ranking Questions (4 options) --}}
                                                @elseif($question->option_d_label)
                                                <div class="ranking-options-group" data-question-id="{{ $question->id }}">
                                                    <div class="ranking-option" data-option="A">{{ $question->option_a_label }}</div>
                                                    <div class="ranking-option" data-option="B">{{ $question->option_b_label }}</div>
                                                    <div class="ranking-option" data-option="C">{{ $question->option_c_label }}</div>
                                                    <div class="ranking-option" data-option="D">{{ $question->option_d_label }}</div>
                                                        </div>
                                                {{-- Hidden fields to store ranking --}}
                                                <input type="hidden" name="questions[{{ $question->id }}][A]" class="ranking-input-a" value="4">
                                                <input type="hidden" name="questions[{{ $question->id }}][B]" class="ranking-input-b" value="4">
                                                <input type="hidden" name="questions[{{ $question->id }}][C]" class="ranking-input-c" value="4">
                                                <input type="hidden" name="questions[{{ $question->id }}][D]" class="ranking-input-d" value="4">
                                                <input type="hidden" class="ranking-complete-check" value="" required>

                                            {{-- Part 6: Three Choice Questions (A/B/C) --}}
                                                @elseif($question->option_c_label)
                                                    <div class="form-check">
                                                        <input class="form-check-input part6-radio" 
                                                               type="radio" 
                                                               name="questions[{{ $question->id }}]" 
                                                               id="q{{ $question->id }}_a" 
                                                               value="A"
                                                               data-question-id="{{ $question->id }}"
                                                               required>
                                                        <label class="form-check-label" for="q{{ $question->id }}_a">
                                                            {{ $question->option_a_label }}
                                                        </label>
                                                    </div>
                                                    <div class="form-check">
                                                        <input class="form-check-input part6-radio" 
                                                               type="radio" 
                                                               name="questions[{{ $question->id }}]" 
                                                               id="q{{ $question->id }}_b" 
                                                               value="B"
                                                               data-question-id="{{ $question->id }}"
                                                               required>
                                                        <label class="form-check-label" for="q{{ $question->id }}_b">
                                                            {{ $question->option_b_label }}
                                                        </label>
                                                    </div>
                                                    <div class="form-check">
                                                        <input class="form-check-input part6-radio" 
                                                               type="radio" 
                                                               name="questions[{{ $question->id }}]" 
                                                               id="q{{ $question->id }}_c" 
                                                               value="C"
                                                               data-question-id="{{ $question->id }}"
                                                               required>
                                                        <label class="form-check-label" for="q{{ $question->id }}_c">
                                                            {{ $question->option_c_label }}
                                                        </label>
                                                    </div>

                                            {{-- Part 2: Binary Choice Questions (2 options) --}}
                                                @else
                                                    <div class="form-check">
                                                        <input class="form-check-input part2-radio" 
                                                               type="radio" 
                                                           name="questions[{{ $question->id }}]" 
                                                               id="q{{ $question->id }}_a" 
                                                               value="A"
                                                               data-question-id="{{ $question->id }}"
                                                               required>
                                                        <label class="form-check-label" for="q{{ $question->id }}_a">
                                                            {{ $question->option_a_label }}
                                                        </label>
                                                    </div>
                                                    <div class="form-check">
                                                        <input class="form-check-input part2-radio" 
                                                               type="radio" 
                                                           name="questions[{{ $question->id }}]" 
                                                               id="q{{ $question->id }}_b" 
                                                               value="B"
                                                               data-question-id="{{ $question->id }}"
                                                               required>
                                                        <label class="form-check-label" for="q{{ $question->id }}_b">
                                                            {{ $question->option_b_label }}
                                                        </label>
                                                    </div>
                                                @endif
                                            </div>
                                        </div>
                                    @endforeach
                                </div>
                            </div>
                        </div>
                    @php $pageOffset++; @endphp
                    @endforeach
                @endforeach
                
                <!-- Congratulations Message -->
                <div class="text-center mb-4" id="completion-message" style="display: none;">
                    <div class="alert alert-success d-inline-block px-5 py-4 border-2" style="border-radius: 15px; box-shadow: 0 4px 12px rgba(0,0,0,0.1);">
                        <h4 class="mb-0"><i class="ri-check-circle-line me-2"></i>Congratulations, you are done</h4>
                    </div>
                </div>

                <!-- Navigation Buttons -->
                <div class="paradigm-navigation">
                    <div class="d-flex justify-content-between align-items-center">
                        <button type="button" class="btn" id="paradigm-prev-btn" style="display: none; background-color: #044f54; color: white; border-color: #044f54;">
                            <i class="ri-arrow-left-line me-2"></i>Previous
                        </button>
                        <button type="button" class="btn" id="paradigm-next-btn" style="display: none; background-color: #044f54; color: white; border-color: #044f54;">
                            Next<i class="ri-arrow-right-line ms-2"></i>
                        </button>
                        <button type="submit" class="btn btn-danger" id="paradigm-submit-btn" style="display: none;">
                            <i class="ri-download-line me-2"></i>Submit And Download Your Report
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Terms and Conditions Modal -->
    <div class="modal fade" id="termsModal" tabindex="-1" aria-labelledby="termsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-dialog-scrollable">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="termsModalLabel">Terms and Conditions</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div id="terms-content">
                        <h4>Terms & Conditions</h4>
                        
                        <h5>Welcome to Paradigm Profiles™</h5>
                        <p>Paradigm Profiles™ is a groundbreaking personal and professional development instrument, developed by the founders of I Seek Help International Pty Ltd.</p>
                        
                        <p>It integrates thinking preferences, personality traits, connection styles, and conflict-resolution strategies into one complete framework, offering profound insights into how you think, connect, and respond to life's challenges.</p>
                        
                        <p>This one-of-a-kind instrument empowers you to elevate every area of your life through deeper self-understanding and actionable insight.</p>
                        
                        <h5>Through your personalized report, you'll uncover strategies to:</h5>
                        <ul>
                            <li>Strengthen communication and relationships.</li>
                            <li>Navigate conflict with confidence and clarity.</li>
                            <li>Align your goals and decisions with your natural strengths.</li>
                            <li>Enhance emotional intelligence and personal growth.</li>
                        </ul>
                        
                        <p>Whether you want to build stronger connections, overcome obstacles, or gain clarity in your direction, Paradigm Profiles™ helps you unlock your full potential.</p>
                        
                        <h5>How Paradigm Profiles™ Guides You</h5>
                        <ul>
                            <li>✅ Uncover your unique strengths and thought patterns.</li>
                            <li>✅ Communicate clearly and effectively, in life and work.</li>
                            <li>✅ Resolve conflicts with empathy and understanding.</li>
                            <li>✅ Deepen emotional intelligence for better decision-making.</li>
                            <li>✅ Build strong, meaningful relationships.</li>
                            <li>✅ Boost self-confidence and identify growth areas.</li>
                            <li>✅ Align your career with your natural skills.</li>
                            <li>✅ Set and achieve goals with precision and purpose.</li>
                            <li>✅ Embrace change and challenges with resilience.</li>
                        </ul>
                        
                        <h5>Begin Your Journey of Transformation</h5>
                        <p>This isn't just another assessment, it's a personalized instrument designed to guide you toward becoming your best version of yourself.</p>
                        
                        <p>By understanding who you truly are, the Paradigm Profile™ empowers you to create a life aligned with your passions, values, and long-term vision.</p>
                        
                        <p>Get ready to understand and get transformed.</p>
                        
                        <h5>Terms and Conditions for Paradigm Profiles™</h5>
                        <p><strong>Last updated:</strong> Fri, 10 October 2025</p>
                        
                        <p>Welcome to Paradigm Profiles™, a service provided by I Seek Help International Pty Ltd.</p>
                        
                        <p>These Terms and Conditions govern your access to and use of the Paradigm Profile Instrument and its related services.</p>
                        
                        <p>By using our services, you acknowledge and agree to these Terms.</p>
                        
                        <h5>1. Definitions</h5>
                        <p><strong>1.1. Company:</strong> I Seek Help International Pty Ltd, provider of Paradigm Profiles.</p>
                        <p><strong>1.2. User:</strong> Any individual who accesses or uses the Paradigm Profile Instrument.</p>
                        <p><strong>1.3. Service:</strong> The Paradigm Profile Instrument, reports, insights, and related support services.</p>
                        <p><strong>1.4. Intellectual Property:</strong> All proprietary rights including copyrights, trademarks, patents, and trade secrets related to the Paradigm Profiles system.</p>
                        
                        <h5>2. Use of the Service</h5>
                        <p><strong>2.1.</strong> The Paradigm Profile is designed for personal, professional, and educational growth. It is not a clinical diagnostic tool and must not replace professional, medical, or legal advice.</p>
                        <p><strong>2.2.</strong> Results are for informational purposes only and do not conclusively define an individual's personality, abilities, or success potential.</p>
                        <p><strong>2.3.</strong> Users must be at least 18 years of age or have verified parental/legal guardian consent.</p>
                        <p><strong>2.4.</strong> By completing the assessment, users grant the Company permission to process and analyze their responses for the generation of their personalized report.</p>
                        
                        <h5>3. Global Data Protection & Privacy Compliance</h5>
                        <p><strong>3.1. Commitment to Global Privacy Standards</strong></p>
                        <p>The Company complies with worldwide data protection and privacy laws, including but not limited to:</p>
                        <ul>
                            <li>South Africa's Protection of Personal Information Act (POPIA)</li>
                            <li>The European Union General Data Protection Regulation (GDPR)</li>
                            <li>The United Kingdom Data Protection Act 2018 (UK DPA)</li>
                            <li>The California Consumer Privacy Act (CCPA)</li>
                            <li>The Personal Information Protection and Electronic Documents Act (PIPEDA) (Canada)</li>
                            <li>The Privacy Act 1988 (Australia)</li>
                        </ul>
                        <p>The Company is committed to safeguarding your personal information globally, in accordance with the highest international standards.</p>
                        
                        <p><strong>3.2. Information Collected</strong></p>
                        <p>The following personal information may be collected and processed:</p>
                        <ul>
                            <li>Full name and surname</li>
                            <li>Email address</li>
                            <li>Contact number</li>
                            <li>Assessment responses and generated results</li>
                            <li>Device or region-related metadata for account and timezone alignment</li>
                        </ul>
                        
                        <p><strong>3.3. Purpose of Data Collection</strong></p>
                        <p>Your information is used solely for:</p>
                        <ul>
                            <li>Generating and delivering your Paradigm Profile report</li>
                            <li>Providing customer support and assistance</li>
                            <li>Improving the quality, security, and accessibility of our services</li>
                        </ul>
                        
                        <p><strong>3.4. Data Security</strong></p>
                        <p>Robust physical, digital, and procedural safeguards are maintained to prevent unauthorized access, misuse, alteration, or disclosure of personal data.</p>
                        <p>Data access is restricted to authorized personnel under confidentiality obligations.</p>
                        
                        <p><strong>3.5. Your Rights (Global Coverage)</strong></p>
                        <p>Depending on your jurisdiction, you may exercise rights under applicable law, including:</p>
                        <ul>
                            <li>Accessing, updating, or correcting your personal data</li>
                            <li>Requesting deletion ("Right to Be Forgotten")</li>
                            <li>Restricting or objecting to processing</li>
                            <li>Withdrawing consent at any time</li>
                            <li>Portability of your data (where applicable)</li>
                            <li>Lodging a complaint with your local Data Protection Authority or Information Regulator</li>
                        </ul>
                        
                        <p><strong>3.6. Data Sharing</strong></p>
                        <p>Your data will never be sold or traded.</p>
                        <p>It will only be shared with trusted service providers who adhere to equivalent data protection standards, or where required by law.</p>
                        
                        <p><strong>3.7. Data Retention</strong></p>
                        <p>Personal information is retained only for as long as necessary to fulfill its purpose.</p>
                        <p>Afterward, it is permanently deleted or anonymized in compliance with global retention policies.</p>
                        
                        <p>For all privacy inquiries:</p>
                        <p>📧 help@iseekhelpoffice.com</p>
                        
                        <h5>4. Intellectual Property Rights</h5>
                        <p><strong>4.1.</strong> The Paradigm Profile framework, design, methodology, and materials are the exclusive intellectual property of I Seek Help International Pty Ltd.</p>
                        <p><strong>4.2.</strong> Users may not reproduce, distribute, modify, or create derivative works without written consent.</p>
                        <p><strong>4.3.</strong> The names Paradigm Profile™, Paradigm Brain Profile™, Personality Blueprints™, Connection Styles™, Conflict Styles™, Paradigm SERQ™ and Paradigm Balance™ are registered trademarks of I Seek Help International Pty Ltd. Unauthorized use is prohibited.</p>
                        
                        <h5>5. Disclaimers and Limitation of Liability</h5>
                        <p><strong>5.1.</strong> The Company provides the Service "as is," without any guarantees of specific outcomes.</p>
                        <p><strong>5.2.</strong> Users are responsible for how they interpret and apply insights from their reports.</p>
                        <p><strong>5.3.</strong> The Service does not replace professional advice in medical, psychological, financial, or legal contexts.</p>
                        
                        <h5>6. Restrictions on Use</h5>
                        <p>Users may not:</p>
                        <ul>
                            <li>Manipulate or misuse the Paradigm Profile Instrument</li>
                            <li>Sell, distribute, or alter the content</li>
                            <li>Use the Service in deceptive, harmful, or unethical ways</li>
                        </ul>
                        <p>Violation of these Terms may result in suspension or termination of access.</p>
                        
                        <h5>7. Payments, Refunds, and Subscriptions</h5>
                        <p><strong>7.1.</strong> Certain services may require payment, as listed on the official website.</p>
                        <p><strong>7.2.</strong> Payments are non-refundable except where required by consumer protection law.</p>
                        <p><strong>7.3.</strong> Subscriptions may be canceled at any time; however, refunds are not provided for unused portions.</p>
                        
                        <h5>8. Third-Party Links and Services</h5>
                        <p>The Service may include links to third-party websites or tools. The Company bears no responsibility for external content, privacy policies, or practices. Use at your own discretion.</p>
                        
                        <h5>9. Modifications to Terms</h5>
                        <p>The Company reserves the right to update or modify these Terms at any time.</p>
                        <p>Continued use of the Service constitutes acceptance of the latest version.</p>
                        <p>Significant updates will be communicated via the website or registered email.</p>
                        
                        <h5>10. Governing Law and Jurisdiction</h5>
                        <p>These Terms and Conditions are governed by the laws of the User's country or region of residence, and, where applicable, by the following jurisdictions:</p>
                        <ul>
                            <li>The Republic of South Africa, as the home jurisdiction of I Seek Help International Pty Ltd, and</li>
                            <li>The United Kingdom of Great Britain and Northern Ireland, as the registered jurisdiction of I Seek Help International Ltd.</li>
                        </ul>
                        <p>Both entities operate under the shared global brand I Seek Help International, maintaining consistent standards for service, data protection, and professional ethics worldwide.</p>
                        
                        <p>Disputes shall first be resolved amicably through negotiation, and if unresolved, shall proceed to mediation or binding arbitration under internationally recognized dispute resolution frameworks (such as UNCITRAL or ICC rules), prior to any formal legal proceedings.</p>
                        
                        <p>The seat of arbitration shall be determined based on the User's primary residence or at the Company's discretion in either South Africa or the United Kingdom, ensuring fairness and accessibility for all parties.</p>
                        
                        <h5>11. Contact Information</h5>
                        <p><strong>I Seek Help International Pty Ltd</strong><br>
                        <strong>and I Seek Help International Ltd</strong></p>
                        <p>📧 Email: help@iseekhelpoffice.com</p>
                        <p>🌐 Website: https://iseekhelpoffice.com</p>
                        
                        <div class="mt-4 p-3 bg-light rounded">
                            <p class="mb-0"><strong>By using the Paradigm Profile Instrument, you confirm that you have read, understood, and accepted these Terms and Conditions, and that your personal information will be protected in accordance with applicable international privacy laws.</strong></p>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Welcome Back Modal -->
    <div class="modal fade" id="welcomeBackModal" tabindex="-1" aria-labelledby="welcomeBackModalLabel" aria-hidden="true" data-bs-backdrop="static" data-bs-keyboard="false">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header" style="background-color: #044f54; color: white;">
                    <h5 class="modal-title" id="welcomeBackModalLabel">Welcome Back!</h5>
                </div>
                <div class="modal-body text-center py-4">
                    <i class="ri-time-line mb-3" style="font-size: 3rem; color: #044f54;"></i>
                    <p class="lead mb-2">You're almost there!</p>
                    <p>We've saved your progress. Let's finish your assessment.</p>
                </div>
                <div class="modal-footer justify-content-center">
                    <button type="button" class="btn btn-lg text-white" id="resume-assessment-btn" style="background-color: #044f54; min-width: 200px;">
                        Continue Assessment <i class="ri-arrow-right-line ms-2"></i>
                    </button>
                </div>
            </div>
        </div>
    </div>
@endsection
    
@section('scripts')
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Hide preloader when page is fully loaded
        window.addEventListener('load', function() {
            const preloader = document.getElementById('paradigm-preloader');
            if (preloader) {
                preloader.style.opacity = '0';
                setTimeout(() => {
                    preloader.style.display = 'none';
                }, 500);
            }
        });
        
        // Fallback: hide preloader after 3 seconds if load event doesn't fire
        setTimeout(() => {
            const preloader = document.getElementById('paradigm-preloader');
            if (preloader && preloader.style.display !== 'none') {
                preloader.style.opacity = '0';
                setTimeout(() => {
                    preloader.style.display = 'none';
                }, 500);
            }
        }, 3000);

        // --- Auto-Save & Restore Logic ---
        const savedProgress = @json($saved_progress ?? null);
        const form = document.getElementById('paradigm-assessment-form');
        
        if (savedProgress && form) {
            restoreProgress(savedProgress);
        }

        function restoreProgress(data) {
            console.log('Restoring progress...', data);
            
            // Restore Personal Details
            if (data.personal) {
                Object.keys(data.personal).forEach(key => {
                    // Try standard name format
                    let input = form.querySelector(`[name="personal[${key}]"]`);
                    
                    // specific handling for phone if it has a visible/hidden pair
                    if (!input && key === 'phone') {
                         input = form.querySelector(`[name="personal[phone_visible]"]`);
                    }

                    if (input) {
                        if (input.type === 'checkbox') {
                            input.checked = !!data.personal[key];
                        } else {
                            input.value = data.personal[key];
                        }
                        // Trigger change for any listeners
                        input.dispatchEvent(new Event('change', { bubbles: true }));
                        input.dispatchEvent(new Event('input', { bubbles: true }));
                    }
                });
            }

            // Restore Terms and Conditions
            if (data.terms_conditions) {
                const termsInput = form.querySelector('[name="terms_conditions"]');
                if (termsInput) {
                    termsInput.checked = true;
                }
            }

            // Restore Questions
            if (data.questions) {
                Object.keys(data.questions).forEach(qId => {
                    const value = data.questions[qId];
                    
                    if (typeof value === 'object') {
                        // Ranking question (Part 1)
                        // value is { A: "1", B: "2", C: "3", D: "4" }
                        // We need to reconstruct the selection order: [Option with rank 1, Option with rank 2, ...]
                        
                        const sortedOptions = Object.keys(value)
                            .filter(opt => value[opt] && value[opt] <= 3) // Only take ranks 1, 2, 3
                            .sort((a, b) => value[a] - value[b]); // Sort by rank
                        
                        // Find the container for this question
                        // The container is usually .ranking-options-group inside #question-{id}
                        // But the ID might be on the parent div.paradigm-question
                        const questionContainer = document.getElementById(`question-${qId}`);
                        if (questionContainer) {
                            const rankingGroup = questionContainer.querySelector('.ranking-options-group');
                            if (rankingGroup) {
                                // Use the existing helper function to restore visuals!
                                updateRankingVisualsAndInputs(rankingGroup, questionContainer, sortedOptions);
                            }
                        }

                    } else {
                        // Radio button question
                        const radio = form.querySelector(`input[name="questions[${qId}]"][value="${value}"]`);
                        if (radio) {
                            radio.checked = true;
                            radio.dispatchEvent(new Event('change', { bubbles: true })); 
                        }
                    }
                });
            }

            // --- Resume Logic ---
            console.log('Starting Resume Logic...');
            // Calculate the first incomplete page
            let firstIncompletePage = 0;
            const pages = document.querySelectorAll('.paradigm-form-page');
            
            for (let i = 0; i < pages.length; i++) {
                const isValid = validatePageElement(pages[i]);
                console.log(`Page ${i} validation:`, isValid);
                
                if (!isValid) {
                    firstIncompletePage = i;
                    break;
                }
                // If valid, we continue to check the next page.
                firstIncompletePage = i + 1; 
            }
            
            // Cap at max pages
            if (firstIncompletePage >= pages.length) {
                firstIncompletePage = pages.length - 1;
            }

            console.log('First incomplete page calculated:', firstIncompletePage);
            
            // If the first incomplete page is NOT the first page (Personal Details), show modal
            if (firstIncompletePage > 0) {
                console.log('Attempting to show Welcome Back modal');
                try {
                    const modalEl = document.getElementById('welcomeBackModal');
                    if (modalEl && typeof bootstrap !== 'undefined') {
                        const welcomeModal = new bootstrap.Modal(modalEl);
                        welcomeModal.show();
                        console.log('Modal shown');
                        
                        document.getElementById('resume-assessment-btn').addEventListener('click', function() {
                            welcomeModal.hide();
                            currentPage = firstIncompletePage;
                            maxPageReached = firstIncompletePage; // Update maxPageReached on resume
                            showPage(currentPage);
                            window.scrollTo(0, 0);
                        });
                    } else {
                        console.error('Bootstrap modal element or library not found');
                        // Fallback: just go there
                        currentPage = firstIncompletePage;
                        maxPageReached = firstIncompletePage; // Update maxPageReached on resume
                        showPage(currentPage);
                    }
                } catch (e) {
                    console.error('Error showing modal:', e);
                }
            }
        }

        function validatePageElement(pageElement) {
            if (!pageElement) return false;
            
            const requiredFields = pageElement.querySelectorAll('[required]');
            let isValid = true;
            
            requiredFields.forEach(field => {
                if (field.type === 'radio') {
                    const radios = pageElement.querySelectorAll(`input[name="${field.name}"]`);
                    if (![...radios].some(radio => radio.checked)) {
                        isValid = false;
                        console.log('Invalid radio:', field.name);
                    }
                } else if (field.type === 'checkbox') {
                    if (!field.checked) {
                        isValid = false;
                         console.log('Invalid checkbox:', field.id);
                    }
                } else if (field.classList.contains('ranking-complete-check')) {
                     if (!field.value) {
                         isValid = false;
                         console.log('Invalid ranking:', field.name);
                     }
                } else if (!field.value) {
                    isValid = false;
                    console.log('Invalid field:', field.name || field.id);
                }
            });
            return isValid;
        }

        function getFormData() {
            if (!form) return null;
            
            const personalInputs = form.querySelectorAll('[name^="personal["]');
            const personalData = {};
            personalInputs.forEach(input => {
                const match = input.name.match(/personal\[(.*?)\]/);
                if (match && match[1]) {
                    personalData[match[1]] = input.value;
                }
            });

            const questionData = {};
            const questionInputs = form.querySelectorAll('[name^="questions["]');
            questionInputs.forEach(input => {
                const match = input.name.match(/questions\[(\d+)\](?:\[([A-D])\])?/);
                if (match) {
                    const qId = match[1];
                    const option = match[2];

                    if (option) { // Ranking
                         if (!questionData[qId]) questionData[qId] = {};
                         questionData[qId][option] = input.value;
                    } else { // Radio
                        if (input.checked) {
                            questionData[qId] = input.value;
                        }
                    }
                }
            });
            
            // Capture Terms and Conditions
            const termsInput = form.querySelector('[name="terms_conditions"]');
            const termsAccepted = termsInput ? termsInput.checked : false;

            return {
                personal: personalData,
                questions: questionData,
                terms_conditions: termsAccepted
            };
        }

        function saveProgress() {
            const data = getFormData();
            if (!data) return;

            const payload = {
                template_id: form.querySelector('input[name="template_id"]').value,
                progress_data: data,
                _token: document.querySelector('input[name="_token"]').value,
                access_token: form.querySelector('input[name="access_token"]')?.value
            };

            const blob = new Blob([JSON.stringify(payload)], { type: 'application/json' });
            
            // Use sendBeacon for reliability on exit
            if (navigator.sendBeacon) {
                navigator.sendBeacon('{{ route("templates.save-progress") }}', blob);
            } else {
                // Fallback for older browsers
                fetch('{{ route("templates.save-progress") }}', {
                    method: 'POST',
                    body: JSON.stringify(payload),
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': payload._token
                    },
                    keepalive: true
                });
            }
        }

        // Save on visibility change (switching tabs, minimizing window)
        document.addEventListener('visibilitychange', function() {
            if (document.visibilityState === 'hidden') {
                saveProgress();
            }
        });

        // Save on beforeunload (closing tab/window)
        window.addEventListener('beforeunload', function() {
            saveProgress();
        });

        // --- End Auto-Save Logic ---

        const phoneInputField = document.querySelector("#personal_phone");
        let phoneInputInstance = null;
        if (phoneInputField) {
            phoneInputInstance = window.intlTelInput(phoneInputField, {
                initialCountry: "us",
                utilsScript: "https://cdnjs.cloudflare.com/ajax/libs/intl-tel-input/17.0.8/js/utils.js",
            });
        }

        // const form = document.getElementById('paradigm-assessment-form'); // Already defined above
        if (!form) return;

        const pages = document.querySelectorAll('.paradigm-form-page');
        const progressBar = document.getElementById('paradigm-progress-bar');
        const currentPageSpan = document.getElementById('paradigm-current-page');
        const totalPagesSpan = document.getElementById('paradigm-total-pages');
        const submitBtn = document.getElementById('paradigm-submit-btn');
        const startBtn = document.getElementById('paradigm-start-btn');
        
        let currentPage = 0;
        let maxPageReached = 0; // Track the furthest page reached
        const totalPages = pages.length;
        
    if (totalPagesSpan) {
        totalPagesSpan.textContent = totalPages;
    }
        
        function showPage(pageIndex) {
            // Update maxPageReached
            if (pageIndex > maxPageReached) {
                maxPageReached = pageIndex;
            }

            pages.forEach((page, index) => {
            if(page) page.style.display = index === pageIndex ? 'block' : 'none';
        });
        
        if (currentPageSpan) currentPageSpan.textContent = pageIndex + 1;
        if (progressBar) {
            const percentage = Math.round(((pageIndex + 1) / totalPages) * 100);
            progressBar.style.width = percentage + '%';
            const progressText = document.getElementById('paradigm-progress-text');
            if (progressText) progressText.textContent = percentage + '%';
        }
        
        // Show/hide navigation buttons
        const prevBtn = document.getElementById('paradigm-prev-btn');
        const nextBtn = document.getElementById('paradigm-next-btn');
        
        if (prevBtn) prevBtn.style.display = pageIndex > 0 ? 'inline-block' : 'none';
        
        // Next button logic: Only show if we are retracing steps (current page < max reached)
        if (nextBtn) {
            nextBtn.style.display = (pageIndex < maxPageReached && pageIndex < totalPages - 1) ? 'inline-block' : 'none';
        }
        
        // Only show submit button and completion message when on last page AND last question is answered
        const isLastPage = pageIndex === totalPages - 1;
        const isLastQuestionAnswered = isLastPage && window.validateCurrentPage();
        
        if (submitBtn) submitBtn.style.display = isLastQuestionAnswered ? 'block' : 'none';
        
        // Show/hide completion message
        const completionMsg = document.getElementById('completion-message');
        if (completionMsg) completionMsg.style.display = isLastQuestionAnswered ? 'block' : 'none';
    }
    
    window.validateCurrentPage = function() {
            const currentPageElement = pages[currentPage];
        if (!currentPageElement) return false;

            const requiredFields = currentPageElement.querySelectorAll('[required]');
            let isValid = true;
            
            requiredFields.forEach(field => {
            if (field.type === 'radio') {
                const radios = currentPageElement.querySelectorAll(`input[name="${field.name}"]`);
                if (![...radios].some(radio => radio.checked)) {
                    isValid = false;
                }
            } else if (field.type === 'checkbox') {
                if (!field.checked) {
                    isValid = false;
                }
            } else if (!field.value) {
                    isValid = false;
                }
            });
            
            return isValid;
    };


    // --- Ranking Logic Helpers ---
    function triggerChangeEvent(input) {
        if(input) {
            let event = new Event('change', { bubbles: true });
            input.dispatchEvent(event);
        }
    }

    function updateRankingVisualsAndInputs(group, container, currentSelections) {
        group.querySelectorAll('.ranking-option').forEach(opt => {
            opt.classList.remove('selected');
            const badge = opt.querySelector('.ranking-badge');
            if (badge) badge.remove();
        });

        currentSelections.forEach((sel, index) => {
            const selectedOption = group.querySelector(`[data-option="${sel}"]`);
            if (selectedOption) {
                selectedOption.classList.add('selected');
                const newBadge = document.createElement('div');
                newBadge.className = 'ranking-badge';
                newBadge.textContent = index + 1;
                selectedOption.prepend(newBadge);
            }
        });

        const inputs = {
            a: container.querySelector('.ranking-input-a'),
            b: container.querySelector('.ranking-input-b'),
            c: container.querySelector('.ranking-input-c'),
            d: container.querySelector('.ranking-input-d')
        };
        if (inputs.a) { inputs.a.value = 4; triggerChangeEvent(inputs.a); }
        if (inputs.b) { inputs.b.value = 4; triggerChangeEvent(inputs.b); }
        if (inputs.c) { inputs.c.value = 4; triggerChangeEvent(inputs.c); }
        if (inputs.d) { inputs.d.value = 4; triggerChangeEvent(inputs.d); }
        
        currentSelections.forEach((sel, index) => {
            const input = container.querySelector(`.ranking-input-${sel.toLowerCase()}`);
            if (input) {
                input.value = index + 1;
                triggerChangeEvent(input);
            }
        });
        
        const completeCheck = container.querySelector('.ranking-complete-check');
        if (completeCheck) {
            completeCheck.value = currentSelections.length === 3 ? 'true' : '';
        }

        // Auto-select 4th option and auto-advance when 3 options are selected
        if (currentSelections.length === 3) {
            const allOptions = ['A', 'B', 'C', 'D'];
            const remainingOption = allOptions.find(opt => !currentSelections.includes(opt));
            
            if (remainingOption) {
                // Auto-select the 4th option
                const remainingInput = container.querySelector(`.ranking-input-${remainingOption.toLowerCase()}`);
                if (remainingInput) {
                    remainingInput.value = 4;
                    triggerChangeEvent(remainingInput);
                }
                
                // Auto-advance to next question immediately when 3rd option is clicked
                setTimeout(() => {
                    if (window.validateCurrentPage()) {
                        if (currentPage < totalPages - 1) {
                            currentPage++;
                            showPage(currentPage);
                        }
                    }
                }, 500); // Reduced delay to 0.5 seconds for faster auto-advance
            }
        }
    }
    // --- End Ranking Logic Helpers ---


    // Start button event listener
    if (startBtn) {
        startBtn.addEventListener('click', function() {
            if (window.validateCurrentPage()) {
                if (currentPage < totalPages - 1) {
                    currentPage++;
                    showPage(currentPage);
                }
            } else {
                alert('Please fill out all required fields and accept the terms and conditions.');
            }
        });
    }
        
        
    if (form) {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            // On submission, update the original phone input's value with the full international number
            if(phoneInputInstance) {
                 const originalPhoneInput = document.querySelector('input[name="personal[phone]"]');
                 if(originalPhoneInput) {
                    originalPhoneInput.value = phoneInputInstance.getNumber();
                 }
            }

            if (window.validateCurrentPage()) {
                if (submitBtn) {
                    submitBtn.innerHTML = '<div class="spinner-border spinner-border-sm" role="status"><span class="visually-hidden">Loading...</span></div> Generating Report...';
                submitBtn.disabled = true;
                }
                
                const personalInputs = form.querySelectorAll('[name^="personal["]');
                const personalData = {};
                personalInputs.forEach(input => {
                    const match = input.name.match(/personal\[(.*?)\]/);
                    if (match && match[1]) {
                        const key = match[1];
                        personalData[key] = input.value;
                    }
                });

                const questionData = {};
                const questionInputs = form.querySelectorAll('[name^="questions["]');
                questionInputs.forEach(input => {
                    const match = input.name.match(/questions\[(\d+)\](?:\[([A-D])\])?/);
                    if (match) {
                        const qId = match[1];
                        const option = match[2];

                        if (option) { // This is a ranking question (Part 1)
                            if (!questionData[qId]) {
                                questionData[qId] = {};
                            }
                            questionData[qId][option] = input.value;
                        } else { // This is a radio button question (Part 2)
                            if (input.checked) {
                                questionData[qId] = input.value;
                            }
                        }
                    }
                });

                const payload = {
                    template_id: form.querySelector('input[name="template_id"]').value,
                    personal: personalData, // Send as object
                    questions: questionData, // Send as object
                    _token: document.querySelector('input[name="_token"]').value
                };
                
                fetch('{{ route("paradigm.assessment.submit") }}', {
                    method: 'POST',
                    body: JSON.stringify(payload), // Only stringify the final payload
                    headers: {
                        'Content-Type': 'application/json', // Set content type to JSON
                        'Accept': 'application/json',
                        'X-CSRF-TOKEN': payload._token
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        if (submitBtn) {
                            submitBtn.innerHTML = '<i class="ri-check-circle-line me-2"></i>Report Ready!';
                        }
                        
                        // Handle file download via direct navigation (works better on mobile)
                        if (data.download_url) {
                            // Create a temporary link to trigger download
                            const link = document.createElement('a');
                            link.href = data.download_url;
                            link.download = ''; // Browser will infer filename
                            document.body.appendChild(link);
                            link.click();
                            document.body.removeChild(link);
                        }
                        
                        // Redirect to dashboard/completion page
                        if (data.redirect_url) {
                            setTimeout(function() {
                                window.location.href = data.redirect_url;
                            }, 2000);
                        }
                    } else {
                        alert('Error: ' + (data.message || 'An unknown error occurred.'));
                        if (submitBtn) {
                            submitBtn.innerHTML = '<i class="ri-download-line me-2"></i>Submit And Download Your Report';
                            submitBtn.disabled = false;
                        }
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('An error occurred. Please try again.');
                    if (submitBtn) {
                        submitBtn.innerHTML = '<i class="ri-download-line me-2"></i>Submit And Download Your Report';
                        submitBtn.disabled = false;
                    }
                });
            } else {
                alert('Please fill out all required fields.');
            }
        });
    }
    
    showPage(0);

    // Ranking Logic Event Listeners
    const rankingGroups = document.querySelectorAll('.ranking-options-group');

    rankingGroups.forEach(group => {
        const options = group.querySelectorAll('.ranking-option');
        const questionId = group.dataset.questionId;
        const questionContainer = document.getElementById(`question-${questionId}`);
        if (!questionContainer) return;
        
        let selections = [];

        options.forEach(option => {
            option.addEventListener('click', () => {
                const currentOption = option.dataset.option;

                if (selections.includes(currentOption)) {
                    selections = selections.filter(sel => sel !== currentOption);
                } else {
                    if (selections.length < 3) {
                        selections.push(currentOption);
                    }
                }
                updateRankingVisualsAndInputs(group, questionContainer, selections);
            });
        });
    });

    // Auto-advance Logic for All Parts
    const part2Radios = document.querySelectorAll('.part2-radio');
    part2Radios.forEach(radio => {
        radio.addEventListener('change', function() {
            if (this.checked) {
                setTimeout(() => {
                    if (window.validateCurrentPage()) {
                        if (currentPage < totalPages - 1) {
                            currentPage++;
                            showPage(currentPage);
                        } else {
                            // If this is the last page and question is answered, show completion
                            showPage(currentPage);
                        }
                    }
                }, 500);
            }
        });
    });

    // Part 6 (Three Choice) Auto-advance Logic
    const part6Radios = document.querySelectorAll('.part6-radio');
    part6Radios.forEach(radio => {
        radio.addEventListener('change', function() {
            if (this.checked) {
                setTimeout(() => {
                    if (window.validateCurrentPage()) {
                        if (currentPage < totalPages - 1) {
                            currentPage++;
                            showPage(currentPage);
                        } else {
                            // If this is the last page and question is answered, show completion
                            showPage(currentPage);
                        }
                    }
                }, 500);
            }
        });
    });

    // Part 7 (Balance Rating Scale) Auto-advance Logic
    const balanceRadios = document.querySelectorAll('.balance-radio');
    balanceRadios.forEach(radio => {
        radio.addEventListener('change', function() {
            if (this.checked) {
                setTimeout(() => {
                    if (window.validateCurrentPage()) {
                        if (currentPage < totalPages - 1) {
                            currentPage++;
                            showPage(currentPage);
                        } else {
                            // If this is the last page and question is answered, show completion
                            showPage(currentPage);
                        }
                    }
                }, 500);
            }
        });
    });

    // Navigation Buttons
    const prevBtn = document.getElementById('paradigm-prev-btn');
    const nextBtn = document.getElementById('paradigm-next-btn');
    
    if (prevBtn) {
        prevBtn.addEventListener('click', function() {
            if (currentPage > 0) {
                currentPage--;
                showPage(currentPage);
            }
        });
    }
    
    if (nextBtn) {
        nextBtn.addEventListener('click', function() {
            if (window.validateCurrentPage()) {
                if (currentPage < totalPages - 1) {
                    currentPage++;
                    showPage(currentPage);
                }
            } else {
                alert('Please fill out all required fields before continuing.');
            }
        });
    }

    // Terms and Conditions Modal - Content is now static, no AJAX needed
    });
    </script>
@endsection
