@extends('layouts.provider-minimal')

@section('content')
<div class="container">
    <div class="row justify-content-center">
        <div class="col-lg-8 col-md-10">
                <!-- Registration Success Notice -->
                <div class="alert mb-4" style="background-color: #044f54; color: white; border: none;">
                    <h5 style="color: white;"><i class="ti ti-check-circle me-2"></i>{{ __('Registration Successful!') }}</h5>
                    <p class="mb-0" style="color: white;">{{ __('One more step: Please verify your identity to complete the process') }}</p>
                </div>
                
                <div class="card shadow-lg border-0">
                    <div class="card-header text-center py-4" style="background-color: #044f54;">
                        <h3 class="mb-0 text-white">{{ __('Identity Verification') }}</h3>
                        <p class="mb-0 text-white" style="opacity: 0.9;">{{ __('Quick and secure - takes less than 2 minutes') }}</p>
                    </div>
                    
                    <div class="card-body p-4">
                        <!-- Hidden Progress (no visible steps like Didit) -->
                        <div id="hidden-progress" style="display: none;">
                            <span id="step-counter">1</span>
                            <span id="progress-bar">50</span>
                        </div>

                        <!-- ID Document -->
                        <div class="verification-step" id="step-1">
                            <div class="text-center mb-4">
                                <i class="ti ti-id fs-1 text-primary mb-3 d-block"></i>
                                <h3 class="mb-2">{{ __('Verify Your Identity') }}</h3>
                                <p class="text-muted">{{ __('Capture your ID document to get started') }}</p>
                            </div>
                            
                            <!-- ID Document Info Cards - 4 Essential Tips -->
                            <div class="row g-3 mb-4">
                                <div class="col-lg-3 col-md-6 col-6">
                                    <div class="info-card info-card-sm">
                                        <div class="info-icon">
                                            <i class="ti ti-id-badge"></i>
                                        </div>
                                        <h6>Valid ID Document</h6>
                                        <p class="mb-0">Passport, National ID, or Driver's License</p>
                                    </div>
                                </div>
                                <div class="col-lg-3 col-md-6 col-6">
                                    <div class="info-card info-card-sm">
                                        <div class="info-icon">
                                            <i class="ti ti-brightness-up"></i>
                                        </div>
                                        <h6>Good Lighting</h6>
                                        <p class="mb-0">Clear photo with no shadows</p>
                                    </div>
                                </div>
                                <div class="col-lg-3 col-md-6 col-6">
                                    <div class="info-card info-card-sm">
                                        <div class="info-icon">
                                            <i class="ti ti-photo"></i>
                                        </div>
                                        <h6>Image Only</h6>
                                        <p class="mb-0">JPEG, PNG, or GIF (no PDFs)</p>
                                    </div>
                                </div>
                                <div class="col-lg-3 col-md-6 col-6">
                                    <div class="info-card info-card-sm">
                                        <div class="info-icon">
                                            <i class="ti ti-shield-lock"></i>
                                        </div>
                                        <h6>100% Secure</h6>
                                        <p class="mb-0">Only for admin verification</p>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row g-2">
                                <div class="col-md-6 col-12">
                                    <div class="upload-option text-center py-3 px-3 border rounded-3 cursor-pointer" id="camera-option-front">
                                        <div class="upload-icon mb-2">
                                            <i class="ti ti-camera"></i>
                                        </div>
                                        <h6 class="mb-1">{{ __('Take Photo') }}</h6>
                                        <p class="text-muted small mb-0" style="font-size: 0.75rem;">{{ __('Use your camera') }}</p>
                                    </div>
                                </div>
                                <div class="col-md-6 col-12">
                                    <div class="upload-option text-center py-3 px-3 border rounded-3 cursor-pointer" onclick="document.getElementById('id-front-file').click()">
                                        <div class="upload-icon mb-2">
                                            <i class="ti ti-upload"></i>
                                        </div>
                                        <h6 class="mb-1">{{ __('Upload File') }}</h6>
                                        <p class="text-muted small mb-0" style="font-size: 0.75rem;">{{ __('Browse from device') }}</p>
                                    </div>
                                    <input type="file" id="id-front-file" accept="image/jpeg,image/jpg,image/png,image/gif" style="display:none;">
                                </div>
                            </div>

                            <!-- Camera Capture Area (Hidden by default) -->
                            <div id="camera-area-front" style="display:none;">
                                <div class="text-center mb-4 position-relative" style="max-width: 400px; margin: 0 auto;">
                                    <div class="camera-container">
                                        <video id="camera-preview-front" width="100%" height="auto" autoplay style="border-radius: 12px;"></video>
                                        <canvas id="camera-canvas-front" style="display:none;"></canvas>
                                        
                                        <!-- ID Card Positioning Overlay - INSIDE video container -->
                                        <div class="id-card-overlay">
                                            <div class="id-card-frame">
                                                <div class="corner corner-tl"></div>
                                                <div class="corner corner-tr"></div>
                                                <div class="corner corner-bl"></div>
                                                <div class="corner corner-br"></div>
                                                <div class="id-card-text-inline">Position ID within brackets</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="text-center">
                                    <button type="button" class="btn btn-custom-primary btn-lg" id="capture-btn-front">
                                        <i class="ti ti-camera me-2"></i>{{ __('Capture Photo') }}
                                    </button>
                                    <button type="button" class="btn btn-custom-secondary btn-lg ms-2" id="cancel-camera-front">
                                        <i class="ti ti-x me-2"></i>{{ __('Cancel') }}
                                    </button>
                                </div>
                            </div>

                            <!-- Preview Area -->
                            <div id="preview-area-front" style="display:none;">
                                <div class="text-center mb-3">
                                    <img id="preview-img-front" src="" alt="ID Front Preview" style="max-width: 100%; max-height: 400px; border-radius: 8px;">
                                </div>
                                <div class="text-center">
                                    <button type="button" class="btn btn-custom-primary btn-lg" id="confirm-btn-front">
                                        <i class="ti ti-check me-2"></i>{{ __('Confirm & Continue') }}
                                    </button>
                                    <button type="button" class="btn btn-custom-secondary btn-lg ms-2" id="retake-btn-front">
                                        <i class="ti ti-refresh me-2"></i>{{ __('Retake') }}
                                    </button>
                                </div>
                            </div>
                        </div>

                        <!-- Selfie -->
                        <div class="verification-step" id="step-2">
                            <div class="text-center mb-4">
                                <i class="ti ti-user-check fs-1 text-success mb-3 d-block"></i>
                                <h3 class="mb-2">{{ __('Great! Now verify it\'s you') }}</h3>
                                <p class="text-muted">{{ __('Take a quick selfie to complete your verification') }}</p>
                            </div>
                            
                            <!-- Selfie Info Cards - 4 Essential Tips -->
                            <div class="row g-3 mb-4">
                                <div class="col-lg-3 col-md-6 col-6">
                                    <div class="info-card info-card-sm">
                                        <div class="info-icon">
                                            <i class="ti ti-sun"></i>
                                        </div>
                                        <h6>Face the Light</h6>
                                        <p class="mb-0">Good front lighting</p>
                                    </div>
                                </div>
                                <div class="col-lg-3 col-md-6 col-6">
                                    <div class="info-card info-card-sm">
                                        <div class="info-icon">
                                            <i class="ti ti-focus-2"></i>
                                        </div>
                                        <h6>Center Face</h6>
                                        <p class="mb-0">Within the oval frame</p>
                                    </div>
                                </div>
                                <div class="col-lg-3 col-md-6 col-6">
                                    <div class="info-card info-card-sm">
                                        <div class="info-icon">
                                            <i class="ti ti-mood-neutral"></i>
                                        </div>
                                        <h6>Neutral Look</h6>
                                        <p class="mb-0">Natural expression</p>
                                    </div>
                                </div>
                                <div class="col-lg-3 col-md-6 col-6">
                                    <div class="info-card info-card-sm">
                                        <div class="info-icon">
                                            <i class="ti ti-shield-lock"></i>
                                        </div>
                                        <h6>100% Secure</h6>
                                        <p class="mb-0">Only for admin verification</p>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row g-2">
                                <div class="col-md-6 col-12">
                                    <div class="upload-option text-center py-3 px-3 border rounded-3 cursor-pointer" id="camera-option-selfie">
                                        <div class="upload-icon upload-icon-success mb-2">
                                            <i class="ti ti-camera"></i>
                                        </div>
                                        <h6 class="mb-1">{{ __('Take Selfie') }}</h6>
                                        <p class="text-muted small mb-0" style="font-size: 0.75rem;">{{ __('Use your camera') }}</p>
                                    </div>
                                </div>
                                <div class="col-md-6 col-12">
                                    <div class="upload-option text-center py-3 px-3 border rounded-3 cursor-pointer" onclick="document.getElementById('selfie-file').click()">
                                        <div class="upload-icon upload-icon-success mb-2">
                                            <i class="ti ti-upload"></i>
                                        </div>
                                        <h6 class="mb-1">{{ __('Upload Photo') }}</h6>
                                        <p class="text-muted small mb-0" style="font-size: 0.75rem;">{{ __('Browse from device') }}</p>
                                    </div>
                                    <input type="file" id="selfie-file" accept="image/*" capture="user" style="display:none;">
                                </div>
                            </div>

                            <!-- Camera Capture Area -->
                            <div id="camera-area-selfie" style="display:none;">
                                <div class="text-center mb-4 position-relative" style="max-width: 400px; margin: 0 auto;">
                                    <div class="camera-container">
                                        <video id="camera-preview-selfie" width="100%" height="auto" autoplay style="border-radius: 12px;"></video>
                                        <canvas id="camera-canvas-selfie" style="display:none;"></canvas>
                                        
                                        <!-- Liveness Check - Minimal Neon Overlay -->
                                        <div class="face-overlay">
                                            <div class="face-oval">
                                                <!-- Dense AI Scanning Dots Grid -->
                                                <div class="face-dots-grid"></div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="text-center">
                                    <button type="button" class="btn btn-custom-primary btn-lg" id="capture-btn-selfie">
                                        <i class="ti ti-camera me-2"></i>{{ __('Capture Selfie') }}
                                    </button>
                                    <button type="button" class="btn btn-custom-secondary btn-lg ms-2" id="cancel-camera-selfie">
                                        <i class="ti ti-x me-2"></i>{{ __('Cancel') }}
                                    </button>
                                </div>
                            </div>

                            <!-- Preview Area -->
                            <div id="preview-area-selfie" style="display:none;">
                                <div class="text-center mb-3">
                                    <img id="preview-img-selfie" src="" alt="Selfie Preview" style="max-width: 100%; max-height: 400px; border-radius: 8px;">
                                </div>
                                <div class="text-center">
                                    <button type="button" class="btn btn-custom-primary btn-lg" id="confirm-btn-selfie">
                                        <i class="ti ti-check me-2"></i>{{ __('Submit for Review') }}
                                    </button>
                                    <button type="button" class="btn btn-custom-secondary btn-lg ms-2" id="retake-btn-selfie">
                                        <i class="ti ti-refresh me-2"></i>{{ __('Retake') }}
                                    </button>
                                </div>
                            </div>
                        </div>

                        <!-- Loading Indicator -->
                        <div id="upload-progress" style="display:none;">
                            <div class="text-center py-5">
                                <div class="spinner-border text-primary" role="status" style="width: 3rem; height: 3rem;">
                                    <span class="visually-hidden">{{ __('Uploading...') }}</span>
                                </div>
                                <p class="mt-3" id="upload-message">{{ __('Uploading document...') }}</p>
                            </div>
                        </div>
                        
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* Camera Container */
.camera-container {
    position: relative;
    display: inline-block;
    width: 100%;
}

/* Custom Buttons - #044f54 Theme */
.btn-custom-primary {
    background-color: #044f54 !important;
    border-color: #044f54 !important;
    color: #ffffff !important;
}

.btn-custom-primary:hover {
    background-color: #033d42 !important;
    border-color: #033d42 !important;
    box-shadow: 0 4px 12px rgba(4, 79, 84, 0.3);
}

.btn-custom-secondary {
    background-color: #6c757d !important;
    border-color: #6c757d !important;
    color: #ffffff !important;
}

.btn-custom-secondary:hover {
    background-color: #5a6268 !important;
    border-color: #5a6268 !important;
}

/* Modern Upload Options */
.upload-option {
    transition: all 0.3s ease;
    cursor: pointer;
    border: 2px solid #e9ecef !important;
    background: #ffffff;
    position: relative;
    overflow: hidden;
}

.upload-option::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(4, 79, 84, 0.05), transparent);
    transition: left 0.5s ease;
}

.upload-option:hover::before {
    left: 100%;
}

.upload-option:hover {
    border-color: #044f54 !important;
    transform: translateY(-4px);
    box-shadow: 0 8px 24px rgba(4, 79, 84, 0.15);
}

.upload-icon {
    width: 48px;
    height: 48px;
    border-radius: 12px;
    background: #044f54;
    color: white;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
    box-shadow: 0 3px 12px rgba(4, 79, 84, 0.25);
    transition: all 0.3s ease;
}

.upload-option:hover .upload-icon {
    transform: scale(1.05);
    box-shadow: 0 6px 20px rgba(4, 79, 84, 0.4);
}

.upload-icon-success {
    background: #044f54;
    box-shadow: 0 3px 12px rgba(4, 79, 84, 0.25);
}

.upload-option:hover .upload-icon-success {
    box-shadow: 0 6px 20px rgba(4, 79, 84, 0.4);
}

/* Info Cards - Compact & Consistent Across All Devices */
.info-card {
    background: #044f54;
    border: none;
    border-radius: 12px;
    padding: 0.875rem 0.75rem;
    text-align: center;
    height: 100%;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    pointer-events: none; /* Not selectable */
    user-select: none; /* Not selectable */
    box-shadow: 0 2px 6px rgba(4, 79, 84, 0.12);
}

.info-card-sm {
    padding: 0.75rem 0.625rem;
}

.info-icon {
    width: 40px;
    height: 40px;
    border-radius: 10px;
    background: rgba(255, 255, 255, 0.15);
    color: #ffffff;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 20px;
    margin-bottom: 0.5rem;
    backdrop-filter: blur(10px);
}

.info-card h6 {
    font-size: 0.813rem;
    font-weight: 600;
    margin-bottom: 0.25rem;
    color: #ffffff;
    line-height: 1.3;
}

.info-card p {
    font-size: 0.688rem;
    line-height: 1.4;
    color: rgba(255, 255, 255, 0.85);
}

/* Keep same size on all devices */
@media (max-width: 768px) {
    .info-card {
        padding: 0.875rem 0.75rem;
    }
    
    .info-card-sm {
        padding: 0.75rem 0.625rem;
    }
    
    .info-icon {
        width: 40px;
        height: 40px;
        font-size: 20px;
    }
    
    .info-card h6 {
        font-size: 0.813rem;
    }
    
    .info-card p {
        font-size: 0.688rem;
    }
}

@media (max-width: 576px) {
    .info-card {
        padding: 0.875rem 0.75rem;
    }
    
    .info-card-sm {
        padding: 0.75rem 0.625rem;
    }
    
    .info-icon {
        width: 40px;
        height: 40px;
        font-size: 20px;
    }
    
    .info-card h6 {
        font-size: 0.813rem;
    }
    
    .info-card p {
        font-size: 0.688rem;
    }
}
.step-circle {
    transition: all 0.3s;
}
.step-circle.active {
    background-color: #0d6efd !important;
    transform: scale(1.1);
}
.step-circle.completed {
    background-color: #198754 !important;
}

/* Ensure verification steps are properly hidden by default */
.verification-step {
    display: none !important;
}

/* When a step is active, show it */
.verification-step.active {
    display: block !important;
}

/* ========================================
   LIVENESS CHECK - MINIMAL NEON OVERLAY
   ======================================== */
.face-overlay {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    pointer-events: none;
    z-index: 5;
}

/* Thin Teal OVAL Frame - Clean & Minimal - SMALLER */
.face-oval {
    position: relative;
    width: 140px;
    height: 180px;
    border: 2px solid #044f54;
    border-radius: 50%;
    box-shadow: 
        0 0 15px rgba(4, 79, 84, 0.5),
        0 0 30px rgba(4, 79, 84, 0.3),
        inset 0 0 15px rgba(4, 79, 84, 0.1);
    animation: neon-pulse 2s ease-in-out infinite;
    overflow: hidden;
}

@keyframes neon-pulse {
    0%, 100% {
        box-shadow: 
            0 0 20px rgba(4, 79, 84, 0.6),
            0 0 40px rgba(4, 79, 84, 0.3),
            inset 0 0 20px rgba(4, 79, 84, 0.1);
    }
    50% {
        box-shadow: 
            0 0 30px rgba(4, 79, 84, 0.8),
            0 0 60px rgba(4, 79, 84, 0.5),
            inset 0 0 30px rgba(4, 79, 84, 0.2);
    }
}

/* Dense Animated Dots Grid - ULTRA DENSE Mini Dots */
.face-dots-grid {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    border-radius: 50%;
    background-image: 
        radial-gradient(circle, #044f54 0.8px, transparent 0.8px),
        radial-gradient(circle, #044f54 0.8px, transparent 0.8px);
    background-size: 8px 8px;
    background-position: 0 0, 4px 4px;
    opacity: 0.6;
    animation: dots-shimmer 3s ease-in-out infinite;
    pointer-events: none;
}

@keyframes dots-shimmer {
    0%, 100% {
        opacity: 0.4;
    }
    50% {
        opacity: 0.7;
    }
}

/* Hide Text - Removed from Oval */
.face-text-inline {
    display: none;
}

/* ========================================
   ID CARD BRACKET OVERLAY STYLES
   ======================================== */
.id-card-overlay {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    pointer-events: none; /* Allow clicks through overlay */
    z-index: 5; /* Lower z-index so buttons aren't blocked */
}

.id-card-frame {
    position: relative;
    width: 340px;
    height: 220px;
    border: 2px solid rgba(4, 79, 84, 0.8);
    border-radius: 12px;
    box-shadow: 
        0 0 0 9999px rgba(0, 0, 0, 0.6),
        inset 0 0 20px rgba(4, 79, 84, 0.2),
        0 0 30px rgba(4, 79, 84, 0.5);
    animation: pulse-frame 2s ease-in-out infinite;
}

@keyframes pulse-frame {
    0%, 100% {
        transform: scale(1);
        border-color: rgba(4, 79, 84, 0.8);
    }
    50% {
        transform: scale(1.005);
        border-color: rgba(4, 79, 84, 1);
    }
}

/* Corner Brackets */
.corner {
    position: absolute;
    width: 40px;
    height: 40px;
}

.corner::before,
.corner::after {
    content: '';
    position: absolute;
    background: #044f54;
    box-shadow: 0 0 8px rgba(4, 79, 84, 0.6);
}

.corner::before {
    width: 40px;
    height: 5px;
}

.corner::after {
    width: 5px;
    height: 40px;
}

/* Top-Left Corner */
.corner-tl {
    top: -3px;
    left: -3px;
}
.corner-tl::before {
    top: 0;
    left: 0;
}
.corner-tl::after {
    top: 0;
    left: 0;
}

/* Top-Right Corner */
.corner-tr {
    top: -3px;
    right: -3px;
}
.corner-tr::before {
    top: 0;
    right: 0;
}
.corner-tr::after {
    top: 0;
    right: 0;
}

/* Bottom-Left Corner */
.corner-bl {
    bottom: -3px;
    left: -3px;
}
.corner-bl::before {
    bottom: 0;
    left: 0;
}
.corner-bl::after {
    bottom: 0;
    left: 0;
}

/* Bottom-Right Corner */
.corner-br {
    bottom: -3px;
    right: -3px;
}
.corner-br::before {
    bottom: 0;
    right: 0;
}
.corner-br::after {
    bottom: 0;
    right: 0;
}

.id-card-text-inline {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    background: rgba(4, 79, 84, 0.9);
    color: #ffffff;
    padding: 6px 12px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 500;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.3);
    backdrop-filter: blur(10px);
    pointer-events: none;
    white-space: nowrap;
}

/* ========================================
   MODERN DIDIT.ME-STYLE SCAN ANIMATION
   ======================================== */

/* Scanning Grid Overlay */
.scan-grid {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-image: 
        linear-gradient(0deg, transparent 24%, rgba(4, 79, 84, 0.05) 25%, rgba(4, 79, 84, 0.05) 26%, transparent 27%, transparent 74%, rgba(4, 79, 84, 0.05) 75%, rgba(4, 79, 84, 0.05) 76%, transparent 77%, transparent),
        linear-gradient(90deg, transparent 24%, rgba(4, 79, 84, 0.05) 25%, rgba(4, 79, 84, 0.05) 26%, transparent 27%, transparent 74%, rgba(4, 79, 84, 0.05) 75%, rgba(4, 79, 84, 0.05) 76%, transparent 77%, transparent);
    background-size: 50px 50px;
    opacity: 0;
    animation: grid-fade-in 0.3s ease-out forwards;
    pointer-events: none;
    z-index: 10;
}

@keyframes grid-fade-in {
    to {
        opacity: 1;
    }
}

/* Single Scan Line - Perfect & Smooth */
.scan-line {
    position: absolute;
    left: 0;
    width: 100%;
    height: 3px;
    background: linear-gradient(90deg, 
        transparent, 
        rgba(4, 79, 84, 0.4), 
        rgba(4, 79, 84, 1), 
        rgba(4, 79, 84, 0.4), 
        transparent
    );
    box-shadow: 
        0 0 20px rgba(4, 79, 84, 0.8),
        0 0 40px rgba(4, 79, 84, 0.5);
    pointer-events: none;
    z-index: 15;
    will-change: top, opacity;
}

.scan-line-1 {
    animation: smooth-scan 2.2s ease-in-out;
}

@keyframes smooth-scan {
    0% {
        top: 0%;
        opacity: 0;
    }
    10% {
        opacity: 1;
    }
    90% {
        opacity: 1;
    }
    100% {
        top: 100%;
        opacity: 0;
    }
}

/* Particle Effects - Slower & Smoother */
.scan-particles {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    overflow: hidden;
    pointer-events: none;
    z-index: 12;
}

.scan-particle {
    position: absolute;
    width: 2px;
    height: 2px;
    background: rgba(4, 79, 84, 0.7);
    border-radius: 50%;
    box-shadow: 0 0 6px rgba(4, 79, 84, 0.5);
    animation: particle-float 2.5s cubic-bezier(0.25, 0.46, 0.45, 0.94) forwards;
}

@keyframes particle-float {
    0% {
        transform: translateY(0) scale(1) rotate(0deg);
        opacity: 0;
    }
    10% {
        opacity: 0.7;
    }
    50% {
        opacity: 0.9;
    }
    90% {
        opacity: 0.3;
    }
    100% {
        transform: translateY(-120px) scale(0.3) rotate(180deg);
        opacity: 0;
    }
}

/* Scanning Frame Animation - Perfect & Smooth */
.scanning-frame {
    animation: scan-pulse 2.5s ease-in-out !important;
    border-color: rgba(4, 79, 84, 1) !important;
    will-change: box-shadow, transform;
}

@keyframes scan-pulse {
    0% {
        box-shadow: 
            0 0 0 9999px rgba(0, 0, 0, 0.6),
            inset 0 0 20px rgba(4, 79, 84, 0.2),
            0 0 30px rgba(4, 79, 84, 0.5);
        transform: scale(1);
    }
    50% {
        box-shadow: 
            0 0 0 9999px rgba(0, 0, 0, 0.6),
            inset 0 0 50px rgba(4, 79, 84, 0.6),
            0 0 70px rgba(4, 79, 84, 0.8),
            0 0 100px rgba(4, 79, 84, 0.5);
        transform: scale(1.005);
    }
    100% {
        box-shadow: 
            0 0 0 9999px rgba(0, 0, 0, 0.6),
            inset 0 0 20px rgba(4, 79, 84, 0.2),
            0 0 30px rgba(4, 79, 84, 0.5);
        transform: scale(1);
    }
}

/* Progress Fill Effect - Perfect & Smooth */
.scan-progress {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 0%;
    background: linear-gradient(180deg, 
        rgba(4, 79, 84, 0.15) 0%, 
        rgba(4, 79, 84, 0.08) 50%,
        transparent 100%
    );
    animation: progress-fill 2.4s ease-in-out forwards;
    pointer-events: none;
    z-index: 11;
    will-change: height;
}

@keyframes progress-fill {
    0% {
        height: 0%;
    }
    100% {
        height: 100%;
    }
}

/* Flash Effect - Softer & Smoother */
.flash-effect {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: radial-gradient(circle, rgba(255, 255, 255, 0.7) 0%, rgba(4, 79, 84, 0.2) 50%, transparent 100%);
    z-index: 20;
    animation: flash 0.6s cubic-bezier(0.25, 0.46, 0.45, 0.94);
    pointer-events: none;
}

@keyframes flash {
    0% {
        opacity: 0;
        transform: scale(0.95);
    }
    40% {
        opacity: 1;
        transform: scale(1);
    }
    100% {
        opacity: 0;
        transform: scale(1.05);
    }
}

/* Corner Glow Animation - Slower & Smoother */
.scanning-frame .corner::before,
.scanning-frame .corner::after {
    animation: corner-glow 3s cubic-bezier(0.25, 0.46, 0.45, 0.94);
}

@keyframes corner-glow {
    0%, 100% {
        box-shadow: 0 0 6px rgba(4, 79, 84, 0.5);
        background: #044f54;
    }
    20%, 80% {
        box-shadow: 0 0 18px rgba(4, 79, 84, 0.8), 0 0 35px rgba(4, 79, 84, 0.4);
        background: rgba(4, 79, 84, 0.95);
    }
    40%, 60% {
        box-shadow: 0 0 30px rgba(4, 79, 84, 1), 0 0 60px rgba(4, 79, 84, 0.6);
        background: rgba(4, 110, 115, 1);
    }
    50% {
        box-shadow: 0 0 35px rgba(4, 79, 84, 1), 0 0 70px rgba(4, 79, 84, 0.7);
        background: rgba(4, 120, 125, 1);
    }
}

/* Success Checkmark Animation - Perfect & Smooth */
.scan-success {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%) scale(0);
    width: 70px;
    height: 70px;
    background: rgba(4, 79, 84, 0.95);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 25;
    animation: success-pop 0.5s ease-out forwards;
    pointer-events: none;
    box-shadow: 0 0 25px rgba(4, 79, 84, 0.7);
    will-change: transform, opacity;
}

.scan-success::after {
    content: '✓';
    color: white;
    font-size: 40px;
    font-weight: bold;
}

@keyframes success-pop {
    0% {
        transform: translate(-50%, -50%) scale(0);
        opacity: 0;
    }
    60% {
        transform: translate(-50%, -50%) scale(1.1);
        opacity: 1;
    }
    100% {
        transform: translate(-50%, -50%) scale(1);
        opacity: 1;
    }
}

/* Scanning Text Animation - Perfect & Smooth */
.scanning-text {
    position: absolute;
    bottom: 10px;
    left: 50%;
    transform: translateX(-50%);
    background: rgba(4, 79, 84, 0.95);
    color: white;
    padding: 6px 18px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 600;
    backdrop-filter: blur(10px);
    animation: text-pulse 1.5s ease-in-out infinite;
    pointer-events: none;
    z-index: 16;
    box-shadow: 0 2px 10px rgba(4, 79, 84, 0.4);
    will-change: opacity;
}

@keyframes text-pulse {
    0%, 100% {
        opacity: 0.8;
    }
    50% {
        opacity: 1;
    }
}

/* Responsive adjustments - Full Mobile Support - SMALLER OVAL */
@media (max-width: 992px) {
    .face-oval {
        width: 130px;
        height: 170px;
    }
}

@media (max-width: 768px) {
    .face-oval {
        width: 120px;
        height: 160px;
    }
}

@media (max-width: 576px) {
    .face-oval {
        width: 110px;
        height: 150px;
    }
    
    .id-card-frame {
        width: 280px;
        height: 180px;
    }
    
    .corner {
        width: 30px;
        height: 30px;
    }
    
    .corner::before {
        width: 30px;
    }
    
    .corner::after {
        height: 30px;
    }
    
    .id-card-text-inline,
    .face-text-inline {
        font-size: 11px;
        padding: 6px 12px;
    }
}

</style>

@endsection

@section('scripts')
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/toastr@2.1.4/build/toastr.min.js"></script>
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/toastr@2.1.4/build/toastr.min.css">
<script>
/**
 * Provider Identity Verification - Inline Script
 */

let currentStream = null;
let currentStep = 1;
let capturedFiles = {
    idFront: null,
    idBack: null,
    selfie: null
};

$(document).ready(function() {
    console.log('ID Verification JS loaded');
    
    // Setup AJAX to always send CSRF token
    $.ajaxSetup({
        headers: {
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        }
    });
    
    // Force show step 1 initially
    $('.verification-step').removeClass('active');
    $('#step-1').addClass('active');
    currentStep = 1;
    console.log('Initial step set to 1');
    
    initializeVerification();
});

function initializeVerification() {
    // Camera options click handlers
    $('#camera-option-front').on('click', function() {
        console.log('Camera front clicked');
        startCamera('front');
    });
    
    $('#camera-option-selfie').on('click', function() {
        console.log('Camera selfie clicked');
        startCamera('selfie');
    });
    
    // File upload handlers
    $('#id-front-file').on('change', function(e) {
        handleFileSelect(e.target.files[0], 'front');
    });
    
    $('#selfie-file').on('change', function(e) {
        handleFileSelect(e.target.files[0], 'selfie');
    });
    
    // Capture button handlers
    $('#capture-btn-front').on('click', function() {
        capturePhoto('front');
    });
    
    $('#capture-btn-selfie').on('click', function() {
        capturePhoto('selfie');
    });
    
    // Cancel camera handlers
    $('#cancel-camera-front, #cancel-camera-selfie').on('click', function() {
        stopCamera();
        $(this).closest('.verification-step').find('[id^="camera-area"]').hide();
        $(this).closest('.verification-step').find('.upload-option').show();
    });
    
    // Confirm button handlers
    $('#confirm-btn-front').on('click', function() {
        console.log('Confirm front clicked, uploading document...');
        uploadDocument(capturedFiles.idFront, 'id_document', function() {
            console.log('Document uploaded successfully, moving to step 2...');
            moveToStep(2); // Go directly to selfie (step 2 now)
        });
    });
    
    $('#confirm-btn-selfie').on('click', function() {
        uploadSelfie(capturedFiles.selfie);
    });
    
    // Retake button handlers
    $('#retake-btn-front, #retake-btn-selfie').on('click', function() {
        const step = $(this).closest('.verification-step').attr('id').replace('step-', '');
        resetStep(step);
    });
}

/**
 * Start camera for photo capture
 */
function startCamera(type) {
    console.log('Starting camera for:', type);
    const videoElement = document.getElementById(`camera-preview-${type}`);
    
    if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
        toastr.error('Camera not supported on this device. Please use file upload.');
        return;
    }
    
    // Request camera access
    navigator.mediaDevices.getUserMedia({ 
        video: { 
            facingMode: type === 'selfie' ? 'user' : 'environment',
            width: { ideal: 1280 },
            height: { ideal: 720 }
        } 
    })
    .then(function(stream) {
        console.log('Camera access granted');
        currentStream = stream;
        videoElement.srcObject = stream;
        
        // Hide upload options for current step, show camera
        $(`#camera-area-${type}`).show();
        $(`.verification-step:visible .upload-option`).hide();
    })
    .catch(function(err) {
        console.error('Camera access error:', err);
        toastr.error('Unable to access camera. Please use file upload instead or check camera permissions.');
    });
}

/**
 * Stop camera stream
 */
function stopCamera() {
    if (currentStream) {
        currentStream.getTracks().forEach(track => track.stop());
        currentStream = null;
    }
}

/**
 * Capture photo from camera with modern scan animation
 */
function capturePhoto(type) {
    const video = document.getElementById(`camera-preview-${type}`);
    const canvas = document.getElementById(`camera-canvas-${type}`);
    const context = canvas.getContext('2d');
    
    // Trigger perfect modern scan animation
    triggerScanAnimation(type);
    
    // Wait for animation to complete before capturing (2.8 seconds - optimized timing)
    setTimeout(function() {
        // Set canvas size to video size
        canvas.width = video.videoWidth;
        canvas.height = video.videoHeight;
        
        // Draw video frame to canvas
        context.drawImage(video, 0, 0, canvas.width, canvas.height);
        
        // Convert canvas to blob
        canvas.toBlob(function(blob) {
            // Create file from blob
            const file = new File([blob], `${type}-${Date.now()}.jpg`, { type: 'image/jpeg' });
            
            // Store captured file
            if (type === 'front') {
                capturedFiles.idFront = file;
            } else if (type === 'back') {
                capturedFiles.idBack = file;
            } else {
                capturedFiles.selfie = file;
            }
            
            // Show preview
            const reader = new FileReader();
            reader.onload = function(e) {
                $(`#preview-img-${type}`).attr('src', e.target.result);
                $(`#camera-area-${type}`).hide();
                $(`#preview-area-${type}`).show();
                stopCamera();
            };
            reader.readAsDataURL(file);
        }, 'image/jpeg', 0.9);
    }, 2800); // Wait for 2.8-second perfect scan animation to complete
}

/**
 * Perfect Modern Scan Animation - Smooth & Glitch-Free
 */
function triggerScanAnimation(type) {
    const overlaySelector = type === 'front' ? '.id-card-overlay' : '.face-overlay';
    const frameSelector = type === 'front' ? '.id-card-frame' : '.face-oval';
    const $overlay = $(overlaySelector);
    const $frame = $(frameSelector);
    
    // Prevent multiple animations at once
    if ($frame.hasClass('scanning-frame')) {
        return;
    }
    
    // 1. Add scanning frame animation immediately (NO flash effect)
    $frame.addClass('scanning-frame');
    
    // 2. Add scan line - single smooth line
    setTimeout(function() {
        $overlay.append('<div class="scan-line scan-line-1"></div>');
    }, 200);
    
    // 3. Add Progress Fill Effect
    setTimeout(function() {
        $overlay.append('<div class="scan-progress"></div>');
    }, 300);
    
    // 4. Add "Scanning..." Text
    setTimeout(function() {
        $overlay.append('<div class="scanning-text">Analyzing...</div>');
    }, 400);
    
    // 5. Show Success Checkmark
    setTimeout(function() {
        $overlay.append('<div class="scan-success"></div>');
    }, 1800);
    
    // 6. Clean up all animation elements - ensure smooth cleanup
    setTimeout(function() {
        $frame.removeClass('scanning-frame');
        $overlay.find('.scan-line').fadeOut(200, function() { $(this).remove(); });
        $overlay.find('.scan-progress').fadeOut(200, function() { $(this).remove(); });
        $overlay.find('.scanning-text').fadeOut(200, function() { $(this).remove(); });
        $overlay.find('.scan-success').fadeOut(200, function() { $(this).remove(); });
    }, 2600);
}

/**
 * Handle file selection from upload
 */
function handleFileSelect(file, type) {
    if (!file) return;
    
    // Validate file - Images ONLY, NO PDFs
    const maxSize = 2 * 1024 * 1024; // 2MB
    const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif'];
    
    if (file.size > maxSize) {
        toastr.error('Image size must be less than 2MB');
        return;
    }
    
    if (!allowedTypes.includes(file.type)) {
        toastr.error('Only image files (JPEG, PNG, GIF) are allowed. PDF and document files are NOT accepted.');
        return;
    }
    
    // Store file
    if (type === 'front') {
        capturedFiles.idFront = file;
    } else if (type === 'back') {
        capturedFiles.idBack = file;
    } else {
        capturedFiles.selfie = file;
    }
    
    // Show preview
    const reader = new FileReader();
    reader.onload = function(e) {
        $(`#preview-img-${type}`).attr('src', e.target.result);
        $('.upload-option').hide();
        $(`#preview-area-${type}`).show();
    };
    reader.readAsDataURL(file);
}

/**
 * Upload ID document
 */
function uploadDocument(file, documentType, callback) {
    if (!file) {
        toastr.error('Please capture or select a file first');
        return;
    }
    
    const formData = new FormData();
    formData.append('file', file);
    formData.append('document_type', documentType);
    
    $('#upload-progress').show();
    $('#upload-message').text('Uploading document...');
    $('.verification-step').hide();
    
    $.ajax({
        url: '/provider/identity-verification/upload-id',
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function(response) {
            if (response.success) {
                toastr.success(response.message);
                $('#upload-progress').hide();
                
                // Mark step as completed
                $('#step-indicator-1 .step-circle').removeClass('bg-primary bg-secondary').addClass('bg-success completed');
                
                if (callback) callback();
            } else {
                toastr.error(response.message);
                $('#upload-progress').hide();
                $('.verification-step').eq(currentStep - 1).show();
            }
        },
        error: function(xhr) {
            const message = xhr.responseJSON?.message || 'Upload failed. Please try again.';
            toastr.error(message);
            $('#upload-progress').hide();
            $('.verification-step').eq(currentStep - 1).show();
        }
    });
}

/**
 * Upload selfie
 */
function uploadSelfie(file) {
    if (!file) {
        toastr.error('Please capture or select a selfie first');
        return;
    }
    
    const formData = new FormData();
    formData.append('file', file);
    
    $('#upload-progress').show();
    $('#upload-message').text('Uploading selfie...');
    $('.verification-step').hide();
    
    $.ajax({
        url: '/provider/identity-verification/upload-selfie',
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function(response) {
            if (response.success) {
                toastr.success(response.message);
                $('#upload-progress').hide();
                
                // Mark step as completed
                $('#step-indicator-2 .step-circle').removeClass('bg-primary bg-secondary').addClass('bg-success completed');
                
                // Show success message and redirect
                toastr.success('All documents submitted successfully! Your account is under review. You will need to login again to check your verification status.');
                
                setTimeout(function() {
                    // Redirect to home page where they can login again
                    window.location.href = '/';
                }, 3000);
            } else {
                toastr.error(response.message);
                $('#upload-progress').hide();
                $('.verification-step').eq(currentStep - 1).show();
            }
        },
        error: function(xhr) {
            const message = xhr.responseJSON?.message || 'Upload failed. Please try again.';
            toastr.error(message);
            $('#upload-progress').hide();
            $('.verification-step').eq(currentStep - 1).show();
        }
    });
}

/**
 * Move to next step
 */
function moveToStep(stepNumber) {
    // Prevent duplicate calls
    if (currentStep === stepNumber) {
        console.log('Already on step:', stepNumber, '- skipping');
        return;
    }
    
    console.log('=== MOVING TO STEP ===');
    console.log('From step:', currentStep, 'To step:', stepNumber);
    currentStep = stepNumber;
    
    // Hide all steps by removing active class
    $('.verification-step').removeClass('active');
    console.log('All steps hidden');
    
    // Show target step by adding active class
    const targetStep = `#step-${stepNumber}`;
    $(targetStep).addClass('active');
    console.log('Target step shown:', targetStep);
    
    // Ensure upload options are visible in the new step
    $(targetStep).find('.upload-option').show();
    $(targetStep).find('[id^="camera-area"]').hide();
    $(targetStep).find('[id^="preview-area"]').hide();
    
    // Update hidden progress (no visible indicators like Didit)
    $('#step-counter').text(stepNumber);
    const progressPercent = (stepNumber / 2) * 100;
    $('#progress-bar').text(progressPercent);
    
    console.log('=== STEP TRANSITION COMPLETE ===');
}

/**
 * Reset step to initial state
 */
function resetStep(stepType) {
    const type = stepType === '1' ? 'front' : (stepType === '2' ? 'back' : 'selfie');
    
    // Clear captured file
    if (type === 'front') {
        capturedFiles.idFront = null;
    } else if (type === 'back') {
        capturedFiles.idBack = null;
    } else {
        capturedFiles.selfie = null;
    }
    
    // Reset file input
    $(`#${type === 'front' ? 'id-front-file' : (type === 'back' ? 'id-back-file' : 'selfie-file')}`).val('');
    
    // Hide preview, show upload options
    $(`#preview-area-${type}`).hide();
    $('.upload-option').show();
}
</script>
@endsection
