@extends('front')
@section('title', __('Register Account') . ' - ' . $companyName)
@section('description', __('Create your account to get started'))
@section('keywords', __('register, account, sign up'))
@section('content')
<div class="register-account-page" style="padding: 4rem 0 3rem 0; background: #f4f7f9; min-height: auto;">
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-12 col-md-8 col-lg-6">
                <div id="chatbot-container" style="background: #ffffff; border-radius: 18px; box-shadow: 0 20px 40px rgba(4, 79, 84, 0.18); display: flex; flex-direction: column; height: min(450px, 60vh); max-height: 60vh; overflow: hidden; margin-bottom: 2rem;">
                    <div class="chat-header" style="background: linear-gradient(120deg, #044f54 0%, #02353a 100%); color: white; padding: 20px 24px; display: flex; flex-direction: column; gap: 6px;">
                        <h1 style="margin: 0; font-size: 1.5rem; font-weight: 600; color: white;">{{ __('User Registration') }}</h1>
                        <p style="margin: 0; opacity: 0.85; font-size: 0.95rem; color: white;">{{ __('Let us create your account step by step') }}</p>
                    </div>
                    <div class="chat-messages" id="chat-messages" style="flex: 1 1 auto; padding: 20px 24px; overflow-y: auto; overflow-x: hidden; display: flex; flex-direction: column; gap: 14px; min-height: 0;">
                    </div>
                    <div class="chat-input" style="padding: 20px 24px; border-top: 1px solid rgba(4, 79, 84, 0.08); display: flex; gap: 12px; background: rgba(248, 251, 251, 0.9);">
                        <input type="text" id="user-input" placeholder="{{ __('Type your response and press Enter...') }}" autocomplete="off" style="flex: 1; border: 1px solid rgba(4, 79, 84, 0.18); border-radius: 999px; padding: 12px 18px; font-size: 0.98rem; outline: none; transition: border 0.2s, box-shadow 0.2s;">
                        <button id="send-btn" aria-label="Send response" type="button" style="background: #044f54; color: white; border: none; border-radius: 999px; width: 44px; height: 44px; cursor: pointer; font-size: 20px; display: flex; align-items: center; justify-content: center; transition: background 0.2s, transform 0.1s;">&rarr;</button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="toast" id="toast" style="position: fixed; top: 24px; right: 24px; min-width: 240px; padding: 14px 16px; border-radius: 12px; box-shadow: 0 20px 40px rgba(4, 79, 84, 0.18); background: #044f54; color: white; font-size: 0.95rem; display: none; gap: 10px; align-items: center; z-index: 9999;"></div>

<div class="loading-overlay" id="loading-overlay" style="position: fixed; inset: 0; background: rgba(15, 23, 42, 0.35); display: none; align-items: center; justify-content: center; z-index: 50;">
    <div class="spinner" style="width: 72px; height: 72px; border-radius: 50%; border: 6px solid rgba(255, 255, 255, 0.35); border-top-color: white; animation: spin 0.9s linear infinite;"></div>
</div>

<style>
    @keyframes spin {
        to { transform: rotate(360deg); }
    }
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(8px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
    .message {
        max-width: 85%;
        padding: 14px 18px;
        border-radius: 12px;
        font-size: 0.96rem;
        line-height: 1.45;
        word-wrap: break-word;
        box-shadow: 0 10px 20px rgba(15, 23, 42, 0.08);
        animation: fadeInUp 0.24s ease both;
    }
    .bot-message {
        align-self: flex-start;
        background: #e8f4f5;
        color: #0f172a;
        border-bottom-left-radius: 4px;
    }
    .user-message {
        align-self: flex-end;
        background: #2abaa6;
        color: white;
        border-bottom-right-radius: 4px;
    }
    .chat-input input:focus {
        border-color: #2abaa6;
        box-shadow: 0 0 0 3px rgba(42, 186, 166, 0.25);
    }
    .chat-input button:disabled {
        opacity: 0.6;
        cursor: not-allowed;
    }
    .chat-input button:not(:disabled):hover {
        background: #02353a;
    }
    .chat-input button:active {
        transform: scale(0.96);
    }
    #chat-messages {
        scroll-behavior: smooth;
        -webkit-overflow-scrolling: touch;
    }
    #chat-messages::-webkit-scrollbar {
        width: 6px;
    }
    #chat-messages::-webkit-scrollbar-track {
        background: #f1f1f1;
        border-radius: 10px;
    }
    #chat-messages::-webkit-scrollbar-thumb {
        background: #044f54;
        border-radius: 10px;
    }
    #chat-messages::-webkit-scrollbar-thumb:hover {
        background: #02353a;
    }
    .toast.visible {
        display: inline-flex;
    }
    .toast.tone-error {
        background: #d14343;
    }
    .toast.tone-success {
        background: #2abaa6;
    }
    .loading-overlay.visible {
        display: flex;
    }
    /* Ensure footer is visible */
    footer {
        position: relative !important;
        z-index: 1 !important;
        margin-top: 0 !important;
        display: block !important;
        visibility: visible !important;
        opacity: 1 !important;
    }
    .footer-top,
    .footer-bottom {
        display: block !important;
        visibility: visible !important;
    }
    .register-account-page {
        position: relative;
        z-index: 1;
        padding-bottom: 3rem !important;
    }
    body {
        overflow-x: hidden;
    }
</style>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', () => {
    const chatMessages = document.getElementById('chat-messages');
    const userInput = document.getElementById('user-input');
    const sendBtn = document.getElementById('send-btn');
    const toast = document.getElementById('toast');
    const loadingOverlay = document.getElementById('loading-overlay');
    const apiBaseUrl = window.location.origin.replace(/\/+$/, '');

    const state = {
        conversationStep: 0,
        userDetails: {},
        isSubmitting: false,
        registerStatus: null,
        otp: {
            digitLimit: 6,
            expiresAt: null,
            expireMinutes: 10,
            maskedDestination: '',
            verifiedCode: null,
            actualCode: null, // Store the actual OTP code sent for verification
        },
    };

    function sanitizePhone(raw) {
        const digitsOnly = raw.replace(/[^\d]/g, '');
        if (!digitsOnly) {
            return raw.trim();
        }
        return `+${digitsOnly}`;
    }

    const validators = {
        text: (value) => value.trim().length > 0,
        email: (value) => /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value),
        phone: (value) => {
            const digits = value.replace(/[^\d]/g, '');
            return digits.length >= 10 && digits.length <= 15;
        },
        password: (value) => value.trim().length >= 8,
    };

    const conversationFlow = [
        {
            message: () => "{{ __('Welcome! Let us create your account. Please provide me with your first name?') }}",
            key: 'first_name',
            validate: validators.text,
            errorMessage: "{{ __('Please share your first name so I know how to address you.') }}",
        },
        {
            message: () => `{{ __('Nice to meet you') }}, ${state.userDetails.first_name || ''}. {{ __('Please provide me with your last name?') }}`,
            key: 'last_name',
            validate: validators.text,
            errorMessage: "{{ __('I need your surname to finish creating your profile.') }}",
        },
        {
            message: () => "{{ __('Please provide me with a username for your account.') }}",
            key: 'name',
            validate: validators.text,
            errorMessage: "{{ __('Please share the username we should display on your profile.') }}",
        },
        {
            message: () => "{{ __('Which email should we use for your account?') }}",
            key: 'email',
            validate: validators.email,
            errorMessage: "{{ __('Please share a valid email address (e.g. name@example.com).') }}",
        },
        {
            message: () => {
                const email = state.userDetails.email || '{{ __('the email you entered') }}';
                return `{{ __('Please confirm') }} ${email} {{ __('is correct email? (yes / no)') }}`;
            },
            key: 'email_confirmation',
            process(input) {
                const value = input.trim().toLowerCase();
                if (['yes', 'y'].includes(value)) {
                    return true;
                }
                if (['no', 'n'].includes(value)) {
                    addMessage("{{ __('No problem. Please provide the correct email address we should use.') }}");
                    jumpBackToStep('email');
                    return true;
                }
                addMessage("{{ __('Please reply with \'yes\' or \'no\'.') }}");
                return false;
            },
        },
        {
            message: () => "{{ __('Please enter your phone number (WhatsApp numbers only, include the full international code, e.g. +12345678900).') }}",
            key: 'phone_number',
            validate: validators.phone,
            transform: (value) => sanitizePhone(value),
            errorMessage: "{{ __('That number doesn\'t look right. Please include the full number with country code (e.g. +12345678900).') }}",
        },
        {
            message: () => {
                const number = state.userDetails.phone_number || '{{ __('the number you entered') }}';
                return `{{ __('Is') }} ${number} {{ __('the correct WhatsApp number for your OTP? (yes / no)') }}`;
            },
            key: 'phone_confirmation',
            async process(input) {
                const value = input.trim().toLowerCase();
                if (['yes', 'y'].includes(value)) {
                    return await requestOtp();
                }
                if (['no', 'n'].includes(value)) {
                    addMessage("{{ __('Thanks for checking. Please re-enter your phone number (include the full international code, e.g. +12345678900).') }}");
                    jumpBackToStep('phone_number');
                    return true;
                }
                addMessage("{{ __('Please reply with \'yes\' or \'no\'.') }}");
                return false;
            },
        },
        {
            message: () => {
                const digits = state.otp.digitLimit || 6;
                const destination = state.otp.maskedDestination || '{{ __('your WhatsApp number') }}';
                return `Great! I've sent a ${digits}-digit verification code to ${destination}. Please enter it below.`;
            },
            async process(input) {
                const code = input.replace(/\D/g, '');
                const expected = state.otp.digitLimit || 6;
                if (code.length !== expected) {
                    addMessage(`The verification code should be ${expected} digits long. Please check and try again.`);
                    return false;
                }
                return await verifyOtpCode(code);
            },
        },
        {
            message: () => "{{ __('Please create a password (minimum 8 characters).') }}",
            key: 'password',
            validate: validators.password,
            errorMessage: "{{ __('Password must be at least 8 characters long.') }}",
        },
        {
            message: () => {
                return "{{ __('Perfect! Let me create your account now.') }}";
            },
            async process(input) {
                return await completeRegistration();
            },
        },
    ];

    function addMessage(text, isUser = false) {
        const messageDiv = document.createElement('div');
        messageDiv.className = `message ${isUser ? 'user-message' : 'bot-message'}`;
        messageDiv.textContent = text;
        chatMessages.appendChild(messageDiv);
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }

    function showToast(message, tone = 'info') {
        toast.textContent = message;
        toast.className = `toast visible tone-${tone}`;
        setTimeout(() => {
            toast.classList.remove('visible');
        }, 4200);
    }

    function setLoading(isLoading, message = '') {
        state.isSubmitting = isLoading;
        loadingOverlay.classList.toggle('visible', isLoading);
        if (isLoading) {
            loadingOverlay.setAttribute('aria-live', 'polite');
            loadingOverlay.setAttribute('aria-label', message || '{{ __('Processing your request...') }}');
        } else {
            loadingOverlay.removeAttribute('aria-live');
            loadingOverlay.removeAttribute('aria-label');
        }
        sendBtn.disabled = isLoading;
        userInput.disabled = isLoading;
    }

    async function requestOtp() {
        setLoading(true, 'Sending your verification code…');
        addMessage('Sending your verification code…');
        try {
            const formData = new FormData();
            formData.append('first_name', state.userDetails.first_name);
            formData.append('last_name', state.userDetails.last_name);
            formData.append('name', state.userDetails.name);
            formData.append('email', state.userDetails.email);
            formData.append('phone_number', state.userDetails.phone_number);
            formData.append('password', 'temp_password_123'); // Temporary password for OTP request only
            formData.append('terms_policy', '1');
            formData.append('_token', '{{ csrf_token() }}');

            const response = await fetch(`${apiBaseUrl}/userregister`, {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                },
            });

            const data = await response.json();
            
            if (response.ok && data.code === 200) {
                if (data.register_status === "1") {
                    // OTP required - now send it via WhatsApp/SMS
                    state.registerStatus = "1";
                    state.otp.digitLimit = parseInt(data.otp_digit_limit) || state.otp.digitLimit;
                    state.otp.expiresAt = data.expires_at || null;
                    state.otp.expireMinutes = parseInt(data.otp_expire_time?.split(' ')[0]) || state.otp.expireMinutes;
                    
                    // Send OTP via WhatsApp/SMS
                    const otpType = data.otp_type || 'whatsapp';
                    const sendOtpResponse = await fetch(`${apiBaseUrl}/api/sms/sendsms`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'Accept': 'application/json',
                            'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        },
                        body: JSON.stringify({
                            otp_type: otpType,
                            to_number: state.userDetails.phone_number,
                            notification_type: 2,
                            type: 2,
                            user_name: state.userDetails.name || state.userDetails.first_name,
                            otp: data.otp,
                            subject: data.email_subject || '',
                            content: data.email_content || '',
                        }),
                    });

                    if (!sendOtpResponse.ok) {
                        const errorData = await sendOtpResponse.json();
                        const message = errorData.message || 'Failed to send OTP. Please try again.';
                        addMessage(message);
                        showToast(message, 'error');
                        return false;
                    }
                    
                    // Use masked_destination if provided, otherwise create from phone number
                    state.otp.maskedDestination = data.masked_destination || 
                        (data.phone_number ? 
                            `your WhatsApp number ending ${String(data.phone_number).slice(-4)}` : 
                            `your WhatsApp number ending ${String(state.userDetails.phone_number).slice(-4)}`);
                    
                    // Store registration data for later use
                    state.userDetails.registrationData = data;
                    // Store the actual OTP code for verification
                    state.otp.actualCode = data.otp;
                    
                    showToast('Verification code sent!', 'success');
                    return true;
                } else {
                    // OTP not required - shouldn't happen in this flow
                    const message = data.message || 'Unable to send the verification code. Please double-check the number.';
                    addMessage(message);
                    showToast(message, 'error');
                    return false;
                }
            } else {
                const message = data.message || (data.errors ? Object.values(data.errors)[0][0] : 'Unable to send the verification code. Please double-check the number.');
                addMessage(message);
                showToast(message, 'error');
                return false;
            }
        } catch (error) {
            console.error('OTP request error:', error);
            const message = 'I couldn\'t send the verification code right now. Please confirm the number and try again.';
            addMessage(message);
            showToast(message, 'error');
            return false;
        } finally {
            setLoading(false);
        }
    }

    async function attemptRegistration() {
        setLoading(true, '{{ __('Creating your account...') }}');
        try {
            const formData = new FormData();
            formData.append('first_name', state.userDetails.first_name);
            formData.append('last_name', state.userDetails.last_name);
            formData.append('name', state.userDetails.name);
            formData.append('email', state.userDetails.email);
            formData.append('phone_number', state.userDetails.phone_number);
            formData.append('password', state.userDetails.password);
            formData.append('terms_policy', '1');
            formData.append('_token', '{{ csrf_token() }}');

            const response = await fetch(`${apiBaseUrl}/userregister`, {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                },
            });

            const data = await response.json();
            
            if (response.ok && data.code === 200) {
                if (data.register_status === "0") {
                    // No OTP required - user is already logged in
                    addMessage("{{ __('Registration successful! Signing you in...') }}");
                    showToast("{{ __('Account created successfully!') }}", 'success');
                    
                    setTimeout(() => {
                        window.location.href = '/user/dashboard';
                    }, 1500);
                    return true;
                } else if (data.register_status === "1") {
                    // OTP required
                    state.registerStatus = "1";
                    state.otp.digitLimit = parseInt(data.otp_digit_limit) || state.otp.digitLimit;
                    state.otp.expiresAt = data.expires_at || null;
                    state.otp.expireMinutes = parseInt(data.otp_expire_time?.split(' ')[0]) || state.otp.expireMinutes;
                    state.otp.maskedDestination = data.phone_number ? 
                        `{{ __('your WhatsApp number ending') }} ${String(data.phone_number).slice(-4)}` : 
                        '{{ __('your WhatsApp number') }}';
                    
                    // Store registration data for OTP verification
                    state.userDetails.registrationData = data;
                    
                    // Move to OTP verification step
                    addMessage(`{{ __('Great! I have sent a') }} ${state.otp.digitLimit}{{ __('-digit verification code to') }} ${state.otp.maskedDestination}. {{ __('Please enter it below.') }}`);
                    showToast(data.message || '{{ __('Verification code sent!') }}', 'success');
                    return true;
                }
            } else {
                const message = data.message || (data.errors ? Object.values(data.errors)[0][0] : '{{ __('Registration failed. Please try again.') }}');
                addMessage(message);
                showToast(message, 'error');
                return false;
            }
        } catch (error) {
            console.error('Registration error:', error);
            const message = '{{ __('An error occurred. Please try again later.') }}';
            addMessage(message);
            showToast(message, 'error');
            return false;
        } finally {
            setLoading(false);
        }
    }

    async function verifyOtpCode(code) {
        setLoading(true, 'Verifying your code…');
        addMessage('Verifying your code…');
        
        // Validate format - must be 6 digits
        if (code.length !== 6) {
            setLoading(false);
            addMessage('The verification code should be 6 digits long. Please check and try again.');
            return false;
        }

        try {
            // Check if OTP has expired (10 minutes)
            if (state.otp.expiresAt) {
                const expiresAt = new Date(state.otp.expiresAt);
                const now = new Date();
                if (now > expiresAt) {
                    setLoading(false);
                    addMessage('This verification code has expired. Please request a new OTP.');
                    showToast('OTP expired.', 'error');
                    return false;
                }
            }

            // Verify OTP against database - check if it matches what was stored
            // We verify against the actual code sent, which was stored in otp_settings table
            if (!state.otp.actualCode) {
                setLoading(false);
                addMessage('OTP code not found. Please request a new OTP.');
                showToast('OTP not found.', 'error');
                return false;
            }

            // Verify OTP against the actual code that was sent and stored
            if (code !== state.otp.actualCode) {
                setLoading(false);
                addMessage('That code did not match. Please try again or request a new OTP.');
                showToast('Invalid OTP code.', 'error');
                return false;
            }

            // Store the verified code for account creation
            state.otp.verifiedCode = code;
            setLoading(false);
            addMessage('Code verified successfully!');
            showToast('Code verified successfully!', 'success');
            return true;
        } catch (error) {
            console.error('OTP verification error:', error);
            setLoading(false);
            addMessage('I hit a snag verifying that code. Please try entering it again.');
            showToast('Unable to verify the code right now.', 'error');
            return false;
        }
    }

    async function completeRegistration() {
        setLoading(true, 'Creating your account…');
        addMessage('Creating your account…');
        try {
            if (!state.otp.verifiedCode) {
                setLoading(false);
                addMessage('Please verify your OTP code first.');
                showToast('OTP verification required.', 'error');
                return false;
            }

            const regData = state.userDetails.registrationData || {};
            const payload = {
                otp: state.otp.verifiedCode,
                login_type: 'register',
                name: state.userDetails.name || regData.name || `${state.userDetails.first_name || regData.first_name || ''} ${state.userDetails.last_name || regData.last_name || ''}`.trim(),
                phone_number: state.userDetails.phone_number || regData.phone_number,
                email: state.userDetails.email || regData.email,
                password: state.userDetails.password,
                first_name: state.userDetails.first_name || regData.first_name,
                last_name: state.userDetails.last_name || regData.last_name,
                // Include fields that backend expects (even if null, as we removed them from form)
                gender: null,
                dob: null,
                address: null,
                country: null,
                state: null,
                city: null,
                postal_code: null,
            };

            console.log('Sending registration payload:', { ...payload, password: '***' });

            const response = await fetch(`${apiBaseUrl}/verify-otp`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                },
                body: JSON.stringify(payload),
            });

            let data;
            try {
                data = await response.json();
            } catch (e) {
                console.error('Failed to parse response as JSON:', e);
                const text = await response.text();
                console.error('Response text:', text);
                throw new Error('Invalid response from server');
            }

            console.log('Registration response:', data);
            
            if (response.ok) {
                addMessage("Registration successful! Signing you in...");
                showToast('Account created successfully!', 'success');
                
                setTimeout(() => {
                    window.location.href = '/user/dashboard';
                }, 1500);
                return true;
            } else {
                const message = data.error || data.message || `Registration failed (${response.status}). Please try again.`;
                console.error('Registration failed:', message, data);
                addMessage(message);
                showToast(message, 'error');
                setLoading(false);
                return false;
            }
        } catch (error) {
            console.error('Registration error:', error);
            const errorMsg = error.message || 'I encountered an error. Please try again.';
            addMessage(errorMsg);
            showToast(errorMsg, 'error');
            setLoading(false);
            return false;
        }
    }

    function processInput(input) {
        if (state.isSubmitting) return;

        const currentStep = conversationFlow[state.conversationStep];
        if (!currentStep) {
            return;
        }

        const inputValue = input.trim();

        if (currentStep.process) {
            const processResult = currentStep.process(inputValue);
            // Handle both sync and async returns
            if (processResult && typeof processResult.then === 'function') {
                // It's a Promise
                processResult.then(result => {
                    if (result) {
                        moveToNextStep();
                    }
                }).catch(err => {
                    console.error('Error processing input:', err);
                    showToast("{{ __('An error occurred. Please try again.') }}", 'error');
                });
            } else {
                // It's a synchronous boolean
                if (processResult) {
                    moveToNextStep();
                }
            }
            return;
        }

        if (currentStep.validate && !currentStep.validate(inputValue)) {
            addMessage(currentStep.errorMessage || "{{ __('Invalid input. Please try again.') }}");
            return;
        }

        const value = currentStep.transform ? currentStep.transform(inputValue) : inputValue;
        state.userDetails[currentStep.key] = value;

        moveToNextStep();
    }

    function getStepIndexByKey(key) {
        return conversationFlow.findIndex((step) => step && step.key === key);
    }

    function jumpBackToStep(key) {
        const targetIndex = getStepIndexByKey(key);
        if (targetIndex >= 0) {
            state.conversationStep = targetIndex - 1;
        }
    }

    function moveToNextStep() {
        state.conversationStep++;

        if (state.conversationStep >= conversationFlow.length) {
            submitRegistration();
            return;
        }

        const nextStep = conversationFlow[state.conversationStep];
        if (nextStep.skipIf && nextStep.skipIf()) {
            moveToNextStep();
            return;
        }

        setTimeout(() => {
            addMessage(nextStep.message());
            // If step has process function but no key, auto-execute it (no user input needed)
            if (nextStep.process && !nextStep.key) {
                const processResult = nextStep.process('');
                if (processResult && typeof processResult.then === 'function') {
                    processResult.then(result => {
                        // Registration complete, don't move to next step (will redirect)
                        if (!result) {
                            // If registration failed, user can try again
                            console.log('Registration failed, staying on current step');
                        }
                    }).catch(err => {
                        console.error('Error in auto-process:', err);
                        showToast("{{ __('An error occurred. Please try again.') }}", 'error');
                    });
                }
            }
        }, 300);
    }


    function handleSend() {
        const input = userInput.value.trim();
        if (!input || state.isSubmitting) return;

        addMessage(input, true);
        userInput.value = '';

        setTimeout(() => {
            processInput(input);
        }, 300);
    }

    sendBtn.addEventListener('click', handleSend);
    userInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
            handleSend();
        }
    });

    // Start conversation
    setTimeout(() => {
        addMessage(conversationFlow[0].message());
    }, 500);
});
</script>
@endpush
@endsection
