@extends('front')

@section('content')

<!-- Breadcrumb -->
<div class="breadcrumb-bar text-center">
    <div class="container">
        <div class="row">
            <div class="col-md-12 col-12">
                <h2 class="breadcrumb-title mb-2">{{ __('Top Up Wallet') }}</h2>
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb justify-content-center mb-0">
                        <li class="breadcrumb-item"><a href="{{ route('home') }}"><i class="ti ti-home-2"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('user.wallet') }}">{{ __('Wallet') }}</a></li>
                        <li class="breadcrumb-item active" aria-current="page">{{ __('Top Up') }}</li>
                    </ol>
                </nav>
            </div>
        </div>
        <div class="breadcrumb-bg">
            <img src="/assets/img/bg/breadcrumb-bg-01.png" class="breadcrumb-bg-1" alt="Img">
            <img src="/assets/img/bg/breadcrumb-bg-02.png" class="breadcrumb-bg-2" alt="Img">
        </div>
    </div>
</div>
<!-- /Breadcrumb -->

<!-- Page Wrapper -->
<div class="page-wrapper">
    <div class="content">
        <div class="container">
            <div class="row justify-content-center">
                @include('user.partials.sidebar')
                <div class="col-xl-9 col-lg-8">
                    <!-- Page Header -->
                    <div class="d-flex justify-content-between align-items-center flex-wrap mb-4">
                        <h4>{{ __('Buy Credits') }}</h4>
                        <div>
                            <a href="{{ route('user.wallet') }}" class="btn btn-outline-secondary">
                                <i class="ti ti-arrow-left me-1"></i> {{ __('Back to Credits') }}
                            </a>
                        </div>
                    </div>

                    <div class="row justify-content-center">
                        <div class="col-lg-12">
                            <!-- Credit Package Selection -->
                            <div class="card">
                                <div class="card-header">
                                    <h5 class="card-title mb-0">{{ __('Select Credit Package') }}</h5>
                                </div>
                                <div class="card-body">
                                    <div id="credit-packages-container">
                                        <div class="text-center py-4">
                                            <div class="spinner-border text-primary" role="status">
                                                <span class="visually-hidden">{{ __('Loading...') }}</span>
                                            </div>
                                            <p class="text-muted mt-2">{{ __('Loading credit packages...') }}</p>
                                        </div>
                                    </div>
                                    <div id="no-packages" style="display: none;">
                                        <div class="alert alert-warning">
                                            <i class="ti ti-alert-triangle me-2"></i>
                                            {{ __('No credit packages available. Please contact support.') }}
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Payment Method Selection -->
                            <div class="card">
                                <div class="card-header">
                                    <h5 class="card-title mb-0">{{ __('Payment Method') }}</h5>
                                </div>
                                <div class="card-body">
                                    <div id="payment-methods-container">
                                        <div class="text-center py-4">
                                            <div class="spinner-border text-primary" role="status">
                                                <span class="visually-hidden">{{ __('Loading...') }}</span>
                                            </div>
                                            <p class="text-muted mt-2">{{ __('Loading payment methods...') }}</p>
                                        </div>
                                    </div>
                                    <div id="no-payment-methods" style="display: none;">
                                        <div class="alert alert-warning">
                                            <i class="ti ti-alert-triangle me-2"></i>
                                            {{ __('No active payment methods available. Please enable Stripe or PayPal in admin settings.') }}
                                        </div>
                                    </div>

                                    <!-- Summary -->
                                    <div class="border-top pt-3 mt-3">
                                        <div class="d-flex justify-content-between mb-2">
                                            <span>{{ __('Selected Package') }}:</span>
                                            <strong><span id="summary_package">No package selected</span></strong>
                                        </div>
                                        <div class="d-flex justify-content-between mb-2">
                                            <span>{{ __('Credits') }}:</span>
                                            <strong><span id="summary_credits">0 Credits</span></strong>
                                        </div>
                                        <div class="d-flex justify-content-between mb-2">
                                            <span>{{ __('Price') }}:</span>
                                            <strong><span id="summary_price">R0.00</span></strong>
                                        </div>
                                        <div class="d-flex justify-content-between border-top pt-2">
                                            <h6>{{ __('Total') }}:</h6>
                                            <h5 class="text-primary mb-0"><span id="summary_total">R0.00</span></h5>
                                        </div>
                                    </div>

                                    <!-- Proceed Button -->
                                    <div class="mt-4">
                                        <button type="button" class="btn btn-lg w-100" id="proceedPayment" style="background-color: #044f54; border-color: #044f54; color: white;" onmouseover="this.style.backgroundColor='#033d41'; this.style.borderColor='#033d41';" onmouseout="this.style.backgroundColor='#044f54'; this.style.borderColor='#044f54';" disabled>
                                            <i class="ti ti-lock me-2"></i>{{ __('Proceed to Payment') }}
                                        </button>
                                    </div>

                                    <!-- Info Alert -->
                                    <div class="alert alert-info mt-3 mb-0">
                                        <i class="ti ti-info-circle me-2"></i>
                                        {{ __('Credits will be added to your wallet immediately after successful payment.') }}
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
    /* NEW LIST VIEW STYLES */
    .credit-package-list-item {
        margin-bottom: 1rem;
        position: relative;
    }

    .credit-package-list-item input[type="radio"] {
        position: absolute;
        opacity: 0;
        pointer-events: none;
    }

    .credit-package-list-item input[type="radio"]:checked + label {
        border-color: #044f54;
        background-color: #f8fbfb;
        box-shadow: 0 4px 15px rgba(4, 79, 84, 0.1);
    }

    .credit-package-list-item label {
        display: flex;
        align-items: center;
        width: 100%;
        padding: 1.5rem;
        border: 1px solid #e5e7eb;
        border-radius: 8px;
        transition: all 0.3s ease;
        cursor: pointer;
        overflow: hidden;
    }
    
    .credit-package-list-item label:hover {
        border-color: #c5dadd;
        box-shadow: 0 2px 8px rgba(0,0,0,0.07);
    }
    
    .package-icon {
        flex-shrink: 0;
        margin-right: 1.5rem;
        font-size: 2rem;
        color: #044f54;
    }

    .package-details {
        flex-grow: 1;
    }

    .package-title {
        font-size: 1.125rem;
        font-weight: 600;
        color: #111827;
        margin-bottom: 0.25rem;
    }

    .package-description {
        font-size: 0.875rem;
        color: #6b7280;
    }

    .package-pricing {
        text-align: right;
        flex-shrink: 0;
        min-width: 150px;
    }

    .package-price {
        font-size: 1.5rem;
        font-weight: 700;
        color: #111827;
    }

    .package-price-per-credit {
        font-size: 0.875rem;
        color: #6b7280;
    }
    
    .package-savings {
        font-size: 0.875rem;
        font-weight: 600;
        color: #16a34a; /* Green */
    }

    .package-badge-area {
        flex-grow: 1;
        display: flex;
        justify-content: center;
        align-items: center;
        padding: 0 1rem;
    }

    /* REMOVED CORNER BADGE STYLES AND REPLACED WITH INLINE BADGES */
    .inline-badge {
        padding: 0.3rem 0.8rem;
        font-size: 0.8rem;
        font-weight: 600;
        border-radius: 9999px; /* pill shape */
        color: #fff;
        line-height: 1;
    }

    .badge-popular {
         background-color: #3b82f6; /* Blue */
    }

    .badge-discount {
        background-color: #16a34a; /* Green */
    }

</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    loadCreditPackages();
    loadPaymentMethods();
    
    // Proceed payment button
    document.getElementById('proceedPayment').addEventListener('click', handlePayment);
});

function loadCreditPackages() {
    fetch('/api/credit-packages', {
        method: 'GET',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        }
    })
    .then(response => response.json())
    .then(packages => {
        const container = document.getElementById('credit-packages-container');
        
        if (!packages || packages.length === 0) {
            document.getElementById('no-packages').style.display = 'block';
            container.style.display = 'none';
            return;
        }
        
        // Store packages globally for later use
        window.creditPackages = packages;
        
        // SORT PACKAGES by credits, from lowest to highest
        packages.sort((a, b) => a.credits - b.credits);
        
        let packagesHTML = '<div class="list-group">';
        
        packages.forEach((package, index) => {
            const hasDiscount = parseFloat(package.discount_percentage) > 0;
            const isMostPopular = package.credits === 60; // Example: mark 60 credits as popular

            let badgeHTML = '';
            if (isMostPopular) {
                badgeHTML = `<span class="inline-badge badge-popular">Most Popular</span>`;
            } else if (hasDiscount) {
                badgeHTML = `<span class="inline-badge badge-discount">${parseFloat(package.discount_percentage).toFixed(0)}% OFF</span>`;
            }

            const savingsAmount = parseFloat(package.discount_amount_zar);
            let savingsHTML = '';
            if (savingsAmount > 0) {
                savingsHTML = `<div class="package-savings">You save R${savingsAmount.toFixed(2)}!</div>`;
            }

            packagesHTML += `
                <div class="credit-package-list-item">
                    <input class="form-check-input" type="radio" name="credit_package" 
                           id="package_${package.id}" value="${package.id}" ${index === 0 ? 'checked' : ''}>
                    <label class="form-check-label" for="package_${package.id}">
                        <div class="package-icon">
                            <i class="ti ti-wallet"></i>
                        </div>
                        <div class="package-details">
                            <div class="package-title">${package.credits} Credits</div>
                            <div class="package-description">Top up your wallet with ${package.credits} credits.</div>
                        </div>
                        <div class="package-badge-area">
                            ${badgeHTML}
                        </div>
                        <div class="package-pricing">
                            <div class="package-price">R${parseFloat(package.final_price_zar).toFixed(2)}</div>
                            <div class="package-price-per-credit">R${(package.final_price_zar / package.credits).toFixed(2)}/credit</div>
                            ${savingsHTML}
                        </div>
                    </label>
                </div>
            `;
        });
        
        packagesHTML += '</div>';
        container.innerHTML = packagesHTML;
        
        // Add event listeners to newly created radio buttons
        document.querySelectorAll('input[name="credit_package"]').forEach(radio => {
            radio.addEventListener('change', function() {
                updateSummary();
            });
        });
        
        updateSummary();
    })
    .catch(error => {
        console.error('Error loading credit packages:', error);
        document.getElementById('credit-packages-container').style.display = 'none';
        document.getElementById('no-packages').style.display = 'block';
    });
}

function loadPaymentMethods() {
    fetch('/api/getpaymentmethod', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        }
    })
    .then(response => response.json())
    .then(methods => {
        const container = document.getElementById('payment-methods-container');
        
        if (!methods || methods.length === 0) {
            document.getElementById('no-payment-methods').style.display = 'block';
            container.style.display = 'none';
            return;
        }
        
        let methodsHTML = '<div class="payment-methods-list">';
        let firstMethod = true;
        
        methods.forEach(method => {
            const methodName = method.label.toLowerCase();
            
            // Skip wallet method and PayPal (as requested)
            if (methodName === 'wallet' || methodName === 'paypal') {
                return;
            }
            
            let icon = '';
            let badgeHTML = '';
            let description = '';
            
            if (methodName === 'stripe') {
                icon = '<i class="ti ti-credit-card fs-24 text-primary me-3"></i>';
                description = '{{ __("Pay with Stripe") }}';
                badgeHTML = '';
            } else if (methodName === 'payfast') {
                icon = '<i class="ti ti-credit-card fs-24 text-primary me-3"></i>';
                description = '{{ __("Pay with PayFast") }}';
                badgeHTML = '';
            } else {
                return; // Skip unsupported methods
            }
            
            methodsHTML += `
                <div class="form-check payment-method-card mb-3">
                    <input class="form-check-input" type="radio" name="payment_method" 
                           id="payment_${methodName}" value="${methodName}" ${firstMethod ? 'checked' : ''}>
                    <label class="form-check-label w-100" for="payment_${methodName}">
                        <div class="d-flex align-items-center justify-content-between p-3 border rounded">
                            <div class="d-flex align-items-center">
                                ${icon}
                                <div>
                                    <h6 class="mb-0">${method.payment_type}</h6>
                                    <small class="text-muted">${description}</small>
                                </div>
                            </div>
                            ${badgeHTML}
                        </div>
                    </label>
                </div>
            `;
            firstMethod = false;
        });
        
        methodsHTML += '</div>';
        container.innerHTML = methodsHTML;
        
        // Add event listeners to newly created radio buttons
        document.querySelectorAll('input[name="payment_method"]').forEach(radio => {
            radio.addEventListener('change', function() {
                updateSummary();
            });
        });
        
        updateSummary();
    })
    .catch(error => {
        console.error('Error loading payment methods:', error);
        document.getElementById('payment-methods-container').style.display = 'none';
        document.getElementById('no-payment-methods').style.display = 'block';
    });
}

function handlePayment() {
    const packageRadio = document.querySelector('input[name="credit_package"]:checked');
    const paymentMethodRadio = document.querySelector('input[name="payment_method"]:checked');
    
    if (!packageRadio) {
        toastr.error('{{ __("Please select a credit package") }}');
        return;
    }
    
    if (!paymentMethodRadio) {
        toastr.error('{{ __("Please select a payment method") }}');
        return;
    }
    
    const packageId = packageRadio.value;
    const paymentMethod = paymentMethodRadio.value;
    
    // Get package details from the selected package
    const selectedPackage = window.creditPackages.find(pkg => pkg.id == packageId);
    if (!selectedPackage) {
        toastr.error('{{ __("Invalid package selected") }}');
        return;
    }

    // Process payment via selected gateway
    // Pass both price and credits to the payment processor
    processUserWalletPayment(selectedPackage.final_price_zar, selectedPackage.credits, paymentMethod, selectedPackage);
}

function processUserWalletPayment(priceZAR, credits, paymentMethod, packageData = null) {
    const userId = {{ Auth::id() }};
    const csrfToken = '{{ csrf_token() }}';

    // Step 1: Add wallet amount record
    fetch('/api/addWalletAmount', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json',
            'X-CSRF-TOKEN': csrfToken
        },
        body: JSON.stringify({
            userId: userId,
            amount: priceZAR, // Send the price in ZAR (what they're paying)
            credits: credits, // Send the credits separately
            paymentMethod: paymentMethod
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            const trxId = data.data.transaction_id;
            
            // Step 2: Redirect to payment gateway
            if (paymentMethod === 'stripe') {
                processStripePayment(packageData.final_price_usd, trxId, csrfToken, packageData);
            } else if (paymentMethod === 'payfast') {
                processPayFastPayment(packageData.final_price_zar, trxId, csrfToken, packageData);
            }
        } else {
            toastr.error(data.message || '{{ __("Error adding wallet record") }}');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        toastr.error('{{ __("Error processing payment") }}');
    });
}

function processStripePayment(amount, trxId, csrfToken, packageData) {
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = '{{ route("stripepayment") }}';
    
    const inputs = {
        '_token': csrfToken,
        'amount': amount, // USD amount for Stripe
        'type': 'user',
        'service_amount': amount,
        'trx_id': trxId,
        'usrname': '{{ Auth::user()->name }}',
        'user_id': {{ Auth::id() }},
        'paymenttype': 1,
        'package_id': packageData.id,
        'package_credits': packageData.credits,
        'name': '{{ Auth::user()->name }}',
        'payment_type': 'wallet'
    };
    
    for (const [key, value] of Object.entries(inputs)) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = key;
        input.value = value;
        form.appendChild(input);
    }
    
    document.body.appendChild(form);
    form.submit();
}

function processPayFastPayment(amount, trxId, csrfToken, packageData) {
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = '{{ route("payfastpayment") }}';
    
    const inputs = {
        '_token': csrfToken,
        'amount': packageData.final_price_zar, // Use ZAR amount from package for PayFast
        'trx_id': trxId,
        'type': 'user',
        'usrname': '{{ Auth::user()->name }}',
        'user_id': {{ Auth::id() }},
        'package_id': packageData.id,
        'package_credits': packageData.credits
    };
    
    for (const [key, value] of Object.entries(inputs)) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = key;
        input.value = value;
        form.appendChild(input);
    }
    
    document.body.appendChild(form);
    form.submit();
}

function updateSummary() {
    const packageRadio = document.querySelector('input[name="credit_package"]:checked');
    const paymentMethodRadio = document.querySelector('input[name="payment_method"]:checked');
    
    if (!packageRadio || !paymentMethodRadio) {
        document.getElementById('proceedPayment').disabled = true;
        return;
    }
    
    const packageId = packageRadio.value;
    const selectedPackage = window.creditPackages.find(pkg => pkg.id == packageId);
    const paymentMethod = paymentMethodRadio.value;
    
    if (!selectedPackage) {
        document.getElementById('proceedPayment').disabled = true;
        return;
    }
    
    // Update summary with package details
    document.getElementById('summary_package').textContent = `${selectedPackage.credits} Credits Package`;
    document.getElementById('summary_credits').textContent = `${selectedPackage.credits} Credits`;
    document.getElementById('summary_price').textContent = `R${parseFloat(selectedPackage.final_price_zar).toFixed(2)}`;
    document.getElementById('summary_total').textContent = `R${parseFloat(selectedPackage.final_price_zar).toFixed(2)}`;

    // Enable proceed button
    document.getElementById('proceedPayment').disabled = false;
}
</script>

@endsection
